import { useState, useEffect } from "react";
import { X } from "lucide-react";
import { supabase } from "@/integrations/supabase/client";
import { motion, AnimatePresence } from "framer-motion";

const AppInstallBanner = () => {
  const [visible, setVisible] = useState(false);
  const [settings, setSettings] = useState({
    enabled: false,
    title: "Install App",
    subtitle: "For better experience",
    link: "#",
    image: "",
  });

  useEffect(() => {
    const dismissed = sessionStorage.getItem("app_install_dismissed");
    if (dismissed) return;

    const fetchSettings = async () => {
      const { data } = await supabase
        .from("site_settings")
        .select("key, value")
        .in("key", [
          "app_install_enabled",
          "app_install_title",
          "app_install_subtitle",
          "app_install_link",
          "app_install_image",
        ]);
      if (data) {
        const map: Record<string, string> = {};
        data.forEach((r) => {
          if (r.value) map[r.key] = r.value;
        });
        const enabled = map.app_install_enabled === "true";
        if (enabled) {
          setSettings({
            enabled: true,
            title: map.app_install_title || "Install App",
            subtitle: map.app_install_subtitle || "For better experience",
            link: map.app_install_link || "#",
            image: map.app_install_image || "",
          });
          setVisible(true);
        }
      }
    };
    fetchSettings();
  }, []);

  const handleDismiss = () => {
    setVisible(false);
    sessionStorage.setItem("app_install_dismissed", "true");
    window.dispatchEvent(new Event("app-install-dismissed"));
  };

  return (
    <AnimatePresence>
      {visible && (
        <motion.div
          initial={{ y: 100, opacity: 0 }}
          animate={{ y: 0, opacity: 1 }}
          exit={{ y: 100, opacity: 0 }}
          transition={{ duration: 0.35, ease: "easeOut" }}
          className="fixed bottom-[68px] sm:bottom-0 left-0 right-0 z-30 pointer-events-none"
        >
          <div className="mx-2 mb-1 sm:mx-0 sm:mb-0 rounded-2xl sm:rounded-none overflow-hidden shadow-2xl pointer-events-auto"
            style={{
              background: "linear-gradient(135deg, #1a1a2e 0%, #16213e 50%, #0f3460 100%)",
            }}
          >
            <div className="px-4 py-3 flex items-center justify-between gap-3">
              <div className="flex items-center gap-3 min-w-0">
                {settings.image ? (
                  <img
                    src={settings.image}
                    alt="App Icon"
                    className="h-12 w-12 rounded-2xl object-cover flex-shrink-0 border-2 border-white/10 shadow-lg"
                  />
                ) : (
                  <div className="h-12 w-12 rounded-2xl bg-primary/20 flex items-center justify-center flex-shrink-0 border-2 border-white/10">
                    <span className="text-xl">📱</span>
                  </div>
                )}
                <div className="min-w-0">
                  <p className="text-white text-sm font-bold truncate">{settings.title}</p>
                  <p className="text-white/60 text-[11px] truncate">{settings.subtitle}</p>
                </div>
              </div>
              <div className="flex items-center gap-2.5 flex-shrink-0">
                <a
                  href="/install"
                  className="bg-white text-foreground font-bold text-xs px-5 py-2 rounded-full hover:bg-white/90 transition-colors shadow-md"
                >
                  Install
                </a>
                <button
                  onClick={handleDismiss}
                  className="p-1.5 rounded-full hover:bg-white/10 transition-colors text-white/50 hover:text-white"
                >
                  <X className="h-5 w-5" />
                </button>
              </div>
            </div>
          </div>
        </motion.div>
      )}
    </AnimatePresence>
  );
};

export default AppInstallBanner;
