import { useState, useEffect, ReactNode } from "react";
import { useLocation } from "react-router-dom";
import { supabase } from "@/integrations/supabase/client";
import { useAuth } from "@/contexts/AuthContext";
import { Shield, Loader2 } from "lucide-react";
import { motion } from "framer-motion";

// These routes are always accessible even in maintenance mode
const BYPASS_ROUTES = ["/admin", "/admin-login", "/auth"];

const MaintenanceGuard = ({ children }: { children: ReactNode }) => {
  const { isAdmin, loading: authLoading } = useAuth();
  const location = useLocation();
  const [maintenanceMode, setMaintenanceMode] = useState(false);
  const [maintenanceMessage, setMaintenanceMessage] = useState("সাইটটি রক্ষণাবেক্ষণের জন্য সাময়িকভাবে বন্ধ আছে।");
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const load = async () => {
      const { data } = await supabase.from("site_settings").select("key, value").in("key", ["auth_maintenance_mode", "auth_maintenance_message"]);
      if (data) {
        data.forEach(r => {
          if (r.key === "auth_maintenance_mode" && r.value === "true") setMaintenanceMode(true);
          if (r.key === "auth_maintenance_message" && r.value) setMaintenanceMessage(r.value);
        });
      }
      setLoading(false);
    };
    load();
  }, []);

  if (loading || authLoading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-background">
        <Loader2 className="h-8 w-8 animate-spin text-primary" />
      </div>
    );
  }

  const isBypassRoute = BYPASS_ROUTES.some(r => location.pathname.startsWith(r));

  if (maintenanceMode && !isAdmin && !isBypassRoute) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-primary/5 via-background to-primary/10 px-4">
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          className="max-w-md w-full text-center"
        >
          <div className="bg-card/90 backdrop-blur-md border border-border/50 rounded-2xl p-10 shadow-xl relative overflow-hidden">
            <div className="absolute top-0 left-0 right-0 h-1 bg-gradient-to-r from-primary/60 via-primary to-primary/60" />
            
            <motion.div
              initial={{ scale: 0.5 }}
              animate={{ scale: 1 }}
              transition={{ type: "spring", stiffness: 200, damping: 15 }}
              className="w-20 h-20 rounded-full bg-primary/10 flex items-center justify-center mx-auto mb-6"
            >
              <Shield className="h-10 w-10 text-primary" />
            </motion.div>

            <h1 className="text-2xl font-bold text-foreground mb-3">🔧 রক্ষণাবেক্ষণ চলছে</h1>
            <p className="text-muted-foreground leading-relaxed mb-6">{maintenanceMessage}</p>

            <div className="p-4 rounded-xl bg-muted/50 border border-border">
              <p className="text-xs text-muted-foreground">
                আমরা শীঘ্রই ফিরে আসবো। অনুগ্রহ করে কিছুক্ষণ পর আবার চেষ্টা করুন। 💜
              </p>
            </div>
          </div>
        </motion.div>
      </div>
    );
  }

  return <>{children}</>;
};

export default MaintenanceGuard;
