import { Link } from "react-router-dom";
import { ShoppingCart, Star } from "lucide-react";
import { motion } from "framer-motion";
import type { Product } from "./productsData";
import { parseDurationOption, getPriceRangeFromDurations } from "@/lib/priceUtils";
import { useAuth } from "@/contexts/AuthContext";

export type { Product };

interface ShopProductCardProps {
  product: Product;
  index?: number;
}

const ShopProductCard = ({ product, index = 0 }: ShopProductCardProps) => {
  const { isReseller } = useAuth();
  // Calculate display price - use range from durations if available
  const displayPrice = product.duration && product.duration.length > 0
    ? getPriceRangeFromDurations(product.duration) || product.price
    : product.price;
  const hasDurationPrices = product.duration?.some((d) => parseDurationOption(d).price !== null);

  return (
    <motion.div
      initial={{ opacity: 0, y: 20 }}
      whileInView={{ opacity: 1, y: 0 }}
      viewport={{ once: true, margin: "-50px" }}
      transition={{ duration: 0.4, delay: index * 0.05 }}
    >
      <Link to={`/product/${product.slug || product.id}`} className="group block bg-card rounded-xl sm:rounded-2xl border border-border overflow-hidden transition-all duration-300 hover:-translate-y-1">
        <div className="relative aspect-square bg-gradient-to-br from-muted to-secondary overflow-hidden">
          {product.badges?.includes("sale") && (
            <span className="absolute top-3 left-3 z-10 px-2.5 py-1 text-[10px] font-bold rounded-full bg-destructive text-destructive-foreground">SALE</span>
          )}
          {product.badges?.includes("limited") && (
            <span className="absolute top-3 right-3 z-10 px-2.5 py-1 text-[10px] font-bold rounded-full text-white" style={{ background: `hsl(var(--badge-limited))` }}>LIMITED</span>
          )}
          <img
            src={product.image}
            alt={product.name}
            className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-500"
          />
          <div className="absolute inset-0 bg-gradient-to-t from-foreground/60 via-transparent to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300 flex items-end justify-center pb-4">
            <span className="inline-flex items-center gap-1.5 bg-primary text-primary-foreground font-semibold text-xs px-4 py-2 rounded-full transform translate-y-4 group-hover:translate-y-0 transition-transform duration-300">
              <ShoppingCart className="h-3.5 w-3.5" />
              View Details
            </span>
          </div>
        </div>
        <div className="p-2.5 sm:p-3.5">
          <div className="flex items-center gap-0.5 mb-1.5">
            {[...Array(5)].map((_, i) => (
              <Star key={i} className="h-3 w-3 fill-primary/80 text-primary/80" />
            ))}
          </div>
          <h3 className="font-heading text-xs sm:text-sm font-semibold text-foreground leading-snug mb-1.5 sm:mb-2 line-clamp-1 group-hover:text-primary transition-colors">
            {product.name}
          </h3>
          <div className="flex items-baseline gap-1.5 sm:gap-2 mb-1.5 sm:mb-2 flex-wrap">
            {product.originalPrice && !hasDurationPrices && (
              <span className="text-xs text-muted-foreground line-through">{product.originalPrice}</span>
            )}
            <span className="font-heading font-bold text-primary text-sm sm:text-base">{displayPrice}</span>
            {isReseller && (() => {
              // Show global reseller price, or derive from per-variation prices
              const globalRP = product.resellerPrice;
              const perVarPrices = product.resellerPrices ? Object.values(product.resellerPrices) : [];
              const displayRP = globalRP || (perVarPrices.length > 0 ? (perVarPrices.length === 1 ? perVarPrices[0] : `${perVarPrices[0]} –`) : null);
              return displayRP ? (
                <span className="text-[10px] font-semibold px-1.5 py-0.5 rounded-full bg-accent text-accent-foreground">🏪 {displayRP}</span>
              ) : null;
            })()}
          </div>
          {product.duration && product.duration.length > 0 && (
            <div className="flex flex-wrap gap-1">
              {product.duration.slice(0, 3).map((d) => {
                const { label } = parseDurationOption(d);
                return (
                  <span key={d} className="px-2 py-0.5 text-[10px] font-medium bg-muted rounded-full text-muted-foreground">
                    {label}
                  </span>
                );
              })}
              {product.duration.length > 3 && (
                <span className="px-2 py-0.5 text-[10px] font-medium bg-muted rounded-full text-muted-foreground">
                  +{product.duration.length - 3}
                </span>
              )}
            </div>
          )}
        </div>
      </Link>
    </motion.div>
  );
};

export default ShopProductCard;
