import { useState, useEffect } from "react";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";
import { Plus, Trash2, Save, Loader2, HelpCircle, GripVertical } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { EmptyState } from "@/components/admin/AdminShared";

interface FAQ {
  id: string;
  product_id: string;
  question: string;
  answer: string;
  sort_order: number;
}

interface Product {
  id: string;
  name: string;
}

const AdminFaqPanel = () => {
  const [products, setProducts] = useState<Product[]>([]);
  const [selectedProductId, setSelectedProductId] = useState<string>("");
  const [faqs, setFaqs] = useState<FAQ[]>([]);
  const [loading, setLoading] = useState(false);
  const [saving, setSaving] = useState(false);
  const [newQuestion, setNewQuestion] = useState("");
  const [newAnswer, setNewAnswer] = useState("");
  const { toast } = useToast();

  useEffect(() => {
    supabase.from("products").select("id, name").order("name").then(({ data }) => {
      if (data) setProducts(data);
    });
  }, []);

  useEffect(() => {
    if (!selectedProductId) { setFaqs([]); return; }
    setLoading(true);
    supabase
      .from("product_faqs")
      .select("*")
      .eq("product_id", selectedProductId)
      .order("sort_order")
      .then(({ data }) => {
        if (data) setFaqs(data);
        setLoading(false);
      });
  }, [selectedProductId]);

  const handleAdd = async () => {
    if (!newQuestion.trim() || !newAnswer.trim() || !selectedProductId) {
      toast({ title: "প্রশ্ন ও উত্তর উভয়ই লিখুন", variant: "destructive" });
      return;
    }
    setSaving(true);
    const { data, error } = await supabase.from("product_faqs").insert({
      product_id: selectedProductId,
      question: newQuestion.trim(),
      answer: newAnswer.trim(),
      sort_order: faqs.length,
    }).select().single();
    setSaving(false);
    if (data) {
      setFaqs(prev => [...prev, data]);
      setNewQuestion("");
      setNewAnswer("");
      toast({ title: "FAQ যোগ হয়েছে ✅" });
    }
    if (error) toast({ title: "Error", description: error.message, variant: "destructive" });
  };

  const handleDelete = async (id: string) => {
    const { error } = await supabase.from("product_faqs").delete().eq("id", id);
    if (!error) {
      setFaqs(prev => prev.filter(f => f.id !== id));
      toast({ title: "FAQ ডিলিট হয়েছে 🗑️" });
    }
  };

  const handleUpdateFaq = async (id: string, question: string, answer: string) => {
    const { error } = await supabase.from("product_faqs").update({ question, answer }).eq("id", id);
    if (!error) {
      toast({ title: "FAQ আপডেট হয়েছে ✅" });
    }
  };

  return (
    <div className="space-y-5">
      {/* Product Selector */}
      <div>
        <label className="block text-sm font-semibold text-foreground mb-2">প্রোডাক্ট সিলেক্ট করুন</label>
        <select
          value={selectedProductId}
          onChange={(e) => setSelectedProductId(e.target.value)}
          className="w-full rounded-xl border-2 border-border bg-card px-4 py-2.5 text-sm text-foreground focus:border-primary outline-none"
        >
          <option value="">-- প্রোডাক্ট বাছাই করুন --</option>
          {products.map(p => (
            <option key={p.id} value={p.id}>{p.name}</option>
          ))}
        </select>
      </div>

      {selectedProductId && (
        <>
          {loading ? (
            <div className="flex justify-center py-10">
              <Loader2 className="h-6 w-6 animate-spin text-primary" />
            </div>
          ) : (
            <>
              {/* Existing FAQs */}
              {faqs.length === 0 ? (
                <EmptyState icon={HelpCircle} title="কোনো FAQ নেই" description="এই প্রোডাক্টের জন্য কোনো FAQ যোগ করা হয়নি।" />
              ) : (
                <div className="space-y-3">
                  {faqs.map((faq, index) => (
                    <div key={faq.id} className="bg-card rounded-xl border border-border p-4 space-y-2">
                      <div className="flex items-start gap-2">
                        <span className="text-xs font-bold text-primary bg-primary/10 rounded-full h-6 w-6 flex items-center justify-center shrink-0 mt-0.5">
                          {index + 1}
                        </span>
                        <div className="flex-1 space-y-2">
                          <Input
                            value={faq.question}
                            onChange={(e) => setFaqs(prev => prev.map(f => f.id === faq.id ? { ...f, question: e.target.value } : f))}
                            onBlur={() => handleUpdateFaq(faq.id, faq.question, faq.answer)}
                            placeholder="প্রশ্ন"
                            className="rounded-lg font-semibold text-sm"
                          />
                          <Textarea
                            value={faq.answer}
                            onChange={(e) => setFaqs(prev => prev.map(f => f.id === faq.id ? { ...f, answer: e.target.value } : f))}
                            onBlur={() => handleUpdateFaq(faq.id, faq.question, faq.answer)}
                            placeholder="উত্তর"
                            rows={2}
                            className="rounded-lg text-sm resize-none"
                          />
                        </div>
                        <Button size="sm" variant="destructive" className="h-8 w-8 p-0 shrink-0" onClick={() => handleDelete(faq.id)}>
                          <Trash2 className="h-3.5 w-3.5" />
                        </Button>
                      </div>
                    </div>
                  ))}
                </div>
              )}

              {/* Add New FAQ */}
              <div className="bg-muted/30 rounded-xl border border-dashed border-border p-4 space-y-3">
                <h4 className="text-sm font-bold text-foreground flex items-center gap-2">
                  <Plus className="h-4 w-4 text-primary" /> নতুন FAQ যোগ করুন
                </h4>
                <Input
                  value={newQuestion}
                  onChange={(e) => setNewQuestion(e.target.value)}
                  placeholder="প্রশ্ন লিখুন..."
                  className="rounded-lg"
                />
                <Textarea
                  value={newAnswer}
                  onChange={(e) => setNewAnswer(e.target.value)}
                  placeholder="উত্তর লিখুন..."
                  rows={3}
                  className="rounded-lg resize-none"
                />
                <Button onClick={handleAdd} disabled={saving} className="gap-2">
                  {saving ? <Loader2 className="h-4 w-4 animate-spin" /> : <Plus className="h-4 w-4" />}
                  FAQ যোগ করুন
                </Button>
              </div>
            </>
          )}
        </>
      )}
    </div>
  );
};

export default AdminFaqPanel;
