import { useState, useEffect } from "react";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { SectionCard } from "@/components/admin/AdminShared";
import { Save, Loader2, Eye, Plus, Trash2, Activity, Settings, Zap } from "lucide-react";
import { DEFAULT_EVENTS, clearPixelCache, type PixelEvent } from "@/hooks/usePixel";

const AdminPixelPanel = () => {
  const { toast } = useToast();
  const [saving, setSaving] = useState(false);
  const [loading, setLoading] = useState(true);

  const [pixelEnabled, setPixelEnabled] = useState(false);
  const [pixelId, setPixelId] = useState("");
  const [accessToken, setAccessToken] = useState("");
  const [testEventCode, setTestEventCode] = useState("");
  const [events, setEvents] = useState<PixelEvent[]>(DEFAULT_EVENTS);
  const [customEvents, setCustomEvents] = useState<{ name: string; enabled: boolean }[]>([]);
  const [newCustomEvent, setNewCustomEvent] = useState("");

  useEffect(() => { loadSettings(); }, []);

  const loadSettings = async () => {
    setLoading(true);
    try {
      const { data } = await supabase.from("site_settings").select("key, value").in("key", [
        "pixel_enabled", "pixel_id", "pixel_access_token", "pixel_test_event_code", "pixel_events", "pixel_custom_events",
      ]);
      const map: Record<string, string> = {};
      data?.forEach((r) => { if (r.value) map[r.key] = r.value; });

      if (map.pixel_enabled !== undefined) setPixelEnabled(map.pixel_enabled === "true");
      if (map.pixel_id) setPixelId(map.pixel_id);
      if (map.pixel_access_token) setAccessToken(map.pixel_access_token);
      if (map.pixel_test_event_code) setTestEventCode(map.pixel_test_event_code);
      try { if (map.pixel_events) setEvents(JSON.parse(map.pixel_events)); } catch {}
      try { if (map.pixel_custom_events) setCustomEvents(JSON.parse(map.pixel_custom_events)); } catch {}
    } catch {} finally { setLoading(false); }
  };

  const upsert = async (key: string, value: string) => {
    const { data: existing } = await supabase.from("site_settings").select("id").eq("key", key).maybeSingle();
    if (existing) await supabase.from("site_settings").update({ value, updated_at: new Date().toISOString() }).eq("key", key);
    else await supabase.from("site_settings").insert({ key, value });
  };

  const handleSave = async () => {
    setSaving(true);
    try {
      await Promise.all([
        upsert("pixel_enabled", String(pixelEnabled)),
        upsert("pixel_id", pixelId),
        upsert("pixel_access_token", accessToken),
        upsert("pixel_test_event_code", testEventCode),
        upsert("pixel_events", JSON.stringify(events)),
        upsert("pixel_custom_events", JSON.stringify(customEvents)),
      ]);
      clearPixelCache();
      toast({ title: "✅ Facebook Pixel সেটিংস সেভ হয়েছে!" });
    } catch {
      toast({ title: "সেভ করতে সমস্যা হয়েছে", variant: "destructive" });
    } finally { setSaving(false); }
  };

  const toggleEvent = (key: string) => {
    setEvents((prev) => prev.map((e) => e.key === key ? { ...e, enabled: !e.enabled } : e));
  };

  const toggleCustomEvent = (name: string) => {
    setCustomEvents((prev) => prev.map((e) => e.name === name ? { ...e, enabled: !e.enabled } : e));
  };

  const addCustomEvent = () => {
    const name = newCustomEvent.trim();
    if (!name) return;
    if (customEvents.find((e) => e.name === name)) { toast({ title: "এই নামে ইভেন্ট আছে", variant: "destructive" }); return; }
    setCustomEvents((prev) => [...prev, { name, enabled: true }]);
    setNewCustomEvent("");
  };

  const removeCustomEvent = (name: string) => {
    setCustomEvents((prev) => prev.filter((e) => e.name !== name));
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center py-20">
        <Loader2 className="h-6 w-6 animate-spin text-primary" />
      </div>
    );
  }

  return (
    <div className="space-y-6 max-w-3xl">
      {/* Pixel On/Off */}
      <SectionCard title="Facebook Pixel" description="Facebook Pixel ট্র্যাকিং চালু/বন্ধ করুন" icon={Activity}>
        <div className="flex items-center justify-between p-4 rounded-xl bg-muted/50 border border-border">
          <div className="flex items-center gap-3">
            <div className="h-10 w-10 rounded-xl bg-blue-500/10 flex items-center justify-center">
              <svg className="h-5 w-5" viewBox="0 0 24 24" fill="#1877F2">
                <path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z" />
              </svg>
            </div>
            <div>
              <p className="font-medium text-sm">Facebook Pixel ট্র্যাকিং</p>
              <p className="text-xs text-muted-foreground">চালু করলে কাস্টমারদের সকল কার্যকলাপ ট্র্যাক হবে</p>
            </div>
          </div>
          <Switch checked={pixelEnabled} onCheckedChange={setPixelEnabled} />
        </div>
      </SectionCard>

      {/* Credentials */}
      <SectionCard title="Pixel Credentials" description="Facebook Pixel ID, Access Token এবং Test Event Code দিন" icon={Settings}>
        <div className="space-y-4">
          <div className="space-y-1.5">
            <Label className="text-xs font-medium">Pixel ID / Dataset ID</Label>
            <Input value={pixelId} onChange={(e) => setPixelId(e.target.value)} placeholder="123456789012345" className="rounded-xl" />
          </div>
          <div className="space-y-1.5">
            <Label className="text-xs font-medium">Access Token</Label>
            <Input type="password" value={accessToken} onChange={(e) => setAccessToken(e.target.value)} placeholder="EAAxxxxxxx..." className="rounded-xl" />
            <p className="text-[11px] text-muted-foreground">Conversions API-এর জন্য ব্যবহার হবে (ঐচ্ছিক)</p>
          </div>
          <div className="space-y-1.5">
            <Label className="text-xs font-medium">Test Event Code</Label>
            <Input value={testEventCode} onChange={(e) => setTestEventCode(e.target.value)} placeholder="TEST12345" className="rounded-xl" />
            <p className="text-[11px] text-muted-foreground">Facebook Events Manager থেকে টেস্ট ইভেন্ট কোড দিন</p>
          </div>
        </div>
      </SectionCard>

      {/* Standard Events */}
      <SectionCard title="স্ট্যান্ডার্ড ইভেন্টস" description="প্রতিটি ইভেন্ট আলাদাভাবে Enable/Disable করুন" icon={Zap}>
        <div className="space-y-3">
          {events.map((ev) => (
            <div key={ev.key} className="flex items-center justify-between p-3 rounded-xl bg-muted/50 border border-border">
              <div>
                <p className="font-medium text-sm">{ev.key}</p>
                <p className="text-xs text-muted-foreground">{ev.label}</p>
              </div>
              <Switch checked={ev.enabled} onCheckedChange={() => toggleEvent(ev.key)} />
            </div>
          ))}
        </div>
      </SectionCard>

      {/* Custom Events */}
      <SectionCard title="কাস্টম ইভেন্টস" description="নিজের ইচ্ছামতো নতুন ইভেন্ট তৈরি করুন" icon={Plus}>
        <div className="space-y-3">
          {customEvents.map((ev) => (
            <div key={ev.name} className="flex items-center justify-between p-3 rounded-xl bg-muted/50 border border-border">
              <div>
                <p className="font-medium text-sm">{ev.name}</p>
              </div>
              <div className="flex items-center gap-2">
                <Switch checked={ev.enabled} onCheckedChange={() => toggleCustomEvent(ev.name)} />
                <Button variant="ghost" size="icon" className="h-8 w-8 text-destructive hover:text-destructive" onClick={() => removeCustomEvent(ev.name)}>
                  <Trash2 className="h-3.5 w-3.5" />
                </Button>
              </div>
            </div>
          ))}

          {customEvents.length === 0 && (
            <p className="text-sm text-muted-foreground text-center py-3">কোনো কাস্টম ইভেন্ট নেই</p>
          )}

          <div className="flex gap-2 mt-2">
            <Input
              value={newCustomEvent}
              onChange={(e) => setNewCustomEvent(e.target.value)}
              placeholder="ইভেন্টের নাম, যেমন: WatchVideo"
              className="rounded-xl"
              onKeyDown={(e) => { if (e.key === "Enter") { e.preventDefault(); addCustomEvent(); } }}
            />
            <Button variant="outline" size="sm" className="gap-1.5 rounded-xl shrink-0 h-10" onClick={addCustomEvent} disabled={!newCustomEvent.trim()}>
              <Plus className="h-4 w-4" /> যোগ করুন
            </Button>
          </div>
        </div>
      </SectionCard>

      {/* Save Button */}
      <div className="sticky bottom-4 z-10">
        <Button onClick={handleSave} disabled={saving} className="gap-1.5 rounded-xl shadow-lg shadow-primary/20 w-full sm:w-auto">
          {saving ? <Loader2 className="h-4 w-4 animate-spin" /> : <Save className="h-4 w-4" />}
          {saving ? "সেভ হচ্ছে..." : "সেটিংস সেভ করুন"}
        </Button>
      </div>
    </div>
  );
};

export default AdminPixelPanel;
