import { useState, useEffect } from "react";
import { supabase } from "@/integrations/supabase/client";
import { Button } from "@/components/ui/button";
import { useToast } from "@/hooks/use-toast";
import { EmptyState } from "./AdminShared";
import { Trash2, Loader2, Inbox } from "lucide-react";

const AdminProductRequestsPanel = () => {
  const [requests, setRequests] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const { toast: reqToast } = useToast();

  const fetchRequests = async () => {
    setLoading(true);
    const { data } = await supabase.from("product_requests").select("*").order("created_at", { ascending: false });
    if (data) setRequests(data);
    setLoading(false);
  };

  useEffect(() => { fetchRequests(); }, []);

  const updateStatus = async (id: string, status: string) => {
    await supabase.from("product_requests").update({ status }).eq("id", id);
    reqToast({ title: `✅ স্ট্যাটাস "${status}" আপডেট হয়েছে!` });
    fetchRequests();
  };

  const handleDelete = async (id: string) => {
    await supabase.from("product_requests").delete().eq("id", id);
    reqToast({ title: "✅ রিকোয়েস্ট ডিলিট হয়েছে!" });
    fetchRequests();
  };

  const statusColor = (s: string) =>
    s === "approved" ? "bg-primary/10 text-primary" : s === "rejected" ? "bg-destructive/10 text-destructive" : "bg-amber-500/10 text-amber-600";
  const statusLabel = (s: string) =>
    s === "approved" ? "অনুমোদিত" : s === "rejected" ? "প্রত্যাখ্যাত" : "পেন্ডিং";

  return (
    <div className="space-y-6 max-w-4xl">
      <h2 className="font-heading font-bold text-lg">প্রোডাক্ট রিকোয়েস্ট</h2>

      {loading ? (
        <div className="flex justify-center py-12"><Loader2 className="h-8 w-8 animate-spin text-primary" /></div>
      ) : requests.length === 0 ? (
        <EmptyState icon={Inbox} title="কোনো রিকোয়েস্ট নেই" description="কাস্টমাররা নতুন প্রোডাক্ট রিকোয়েস্ট করলে এখানে দেখা যাবে।" />
      ) : (
        <div className="space-y-3">
          {requests.map(req => (
            <div key={req.id} className="bg-card border border-border rounded-xl p-3 sm:p-4 space-y-2">
              <div className="flex items-start justify-between gap-2">
                <div className="min-w-0">
                  <h4 className="font-semibold text-sm">{req.product_name}</h4>
                  {req.details && <p className="text-xs text-muted-foreground mt-1">{req.details}</p>}
                  <div className="flex flex-wrap gap-2 mt-2 text-[11px] text-muted-foreground">
                    {req.user_email && <span>📧 {req.user_email}</span>}
                    {req.user_phone && <span>📱 {req.user_phone}</span>}
                    <span>📅 {new Date(req.created_at).toLocaleDateString("bn-BD")}</span>
                  </div>
                </div>
                <span className={`text-[10px] px-2 py-0.5 rounded-full font-medium flex-shrink-0 ${statusColor(req.status)}`}>
                  {statusLabel(req.status)}
                </span>
              </div>
              <div className="flex flex-wrap gap-1.5 pt-2 border-t border-border">
                {req.status === "pending" && (
                  <>
                    <Button size="sm" className="h-7 text-xs rounded-lg gap-1" onClick={() => updateStatus(req.id, "approved")}>অনুমোদন</Button>
                    <Button size="sm" variant="outline" className="h-7 text-xs rounded-lg gap-1" onClick={() => updateStatus(req.id, "rejected")}>প্রত্যাখ্যান</Button>
                  </>
                )}
                <Button size="sm" variant="ghost" className="h-7 text-xs rounded-lg text-destructive gap-1" onClick={() => handleDelete(req.id)}>
                  <Trash2 className="h-3 w-3" />ডিলিট
                </Button>
              </div>
            </div>
          ))}
        </div>
      )}
    </div>
  );
};

export default AdminProductRequestsPanel;
