import { useState, useEffect } from "react";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";
import { Check, X, Trash2, Star, Loader2, Eye } from "lucide-react";
import { Button } from "@/components/ui/button";
import { EmptyState } from "@/components/admin/AdminShared";

interface Review {
  id: string;
  product_id: string;
  user_name: string;
  rating: number;
  comment: string | null;
  is_approved: boolean;
  created_at: string;
  product_name?: string;
}

const AdminReviewsPanel = () => {
  const [reviews, setReviews] = useState<Review[]>([]);
  const [loading, setLoading] = useState(true);
  const [filter, setFilter] = useState<"all" | "pending" | "approved">("all");
  const { toast } = useToast();

  const fetchReviews = async () => {
    setLoading(true);
    const { data, error } = await supabase
      .from("product_reviews")
      .select("*")
      .order("created_at", { ascending: false });

    if (data) {
      // Fetch product names
      const productIds = [...new Set(data.map((r: any) => r.product_id))];
      const { data: products } = await supabase
        .from("products")
        .select("id, name")
        .in("id", productIds);

      const productMap: Record<string, string> = {};
      products?.forEach((p: any) => { productMap[p.id] = p.name; });

      setReviews(data.map((r: any) => ({
        ...r,
        product_name: productMap[r.product_id] || "Unknown",
      })));
    }
    if (error) toast({ title: "Error", description: error.message, variant: "destructive" });
    setLoading(false);
  };

  useEffect(() => { fetchReviews(); }, []);

  const handleApprove = async (id: string, approve: boolean) => {
    const { error } = await supabase
      .from("product_reviews")
      .update({ is_approved: approve })
      .eq("id", id);
    if (!error) {
      toast({ title: approve ? "রিভিউ অ্যাপ্রুভ হয়েছে ✅" : "রিভিউ রিজেক্ট হয়েছে" });
      setReviews(prev => prev.map(r => r.id === id ? { ...r, is_approved: approve } : r));
    }
  };

  const handleDelete = async (id: string) => {
    const { error } = await supabase.from("product_reviews").delete().eq("id", id);
    if (!error) {
      toast({ title: "রিভিউ ডিলিট হয়েছে 🗑️" });
      setReviews(prev => prev.filter(r => r.id !== id));
    }
  };

  const filtered = reviews.filter(r => {
    if (filter === "pending") return !r.is_approved;
    if (filter === "approved") return r.is_approved;
    return true;
  });

  const pendingCount = reviews.filter(r => !r.is_approved).length;

  if (loading) {
    return (
      <div className="flex items-center justify-center py-20">
        <Loader2 className="h-8 w-8 animate-spin text-primary" />
      </div>
    );
  }

  return (
    <div className="space-y-4">
      {/* Filter tabs */}
      <div className="flex gap-2 flex-wrap">
        {([
          { key: "all", label: `সব (${reviews.length})` },
          { key: "pending", label: `পেন্ডিং (${pendingCount})` },
          { key: "approved", label: `অ্যাপ্রুভড (${reviews.length - pendingCount})` },
        ] as const).map(f => (
          <button
            key={f.key}
            onClick={() => setFilter(f.key)}
            className={`px-4 py-2 text-sm font-medium rounded-lg transition-all ${
              filter === f.key
                ? "bg-primary text-primary-foreground"
                : "bg-muted text-muted-foreground hover:bg-muted/80"
            }`}
          >
            {f.label}
          </button>
        ))}
      </div>

      {filtered.length === 0 ? (
        <EmptyState icon={Star} title="কোনো রিভিউ নেই" description="কোনো রিভিউ পাওয়া যায়নি" />
      ) : (
        <div className="space-y-3">
          {filtered.map((review) => (
            <div key={review.id} className="bg-card rounded-xl border border-border p-4">
              <div className="flex flex-col sm:flex-row sm:items-start gap-3">
                <div className="flex-1 min-w-0">
                  <div className="flex items-center gap-2 mb-1 flex-wrap">
                    <span className="font-semibold text-sm text-foreground">{review.user_name}</span>
                    <div className="flex items-center gap-0.5">
                      {[1, 2, 3, 4, 5].map(s => (
                        <Star key={s} className={`h-3.5 w-3.5 ${s <= review.rating ? "fill-yellow-400 text-yellow-400" : "text-muted-foreground/30"}`} />
                      ))}
                    </div>
                    <span className={`px-2 py-0.5 text-[10px] font-bold rounded-full ${
                      review.is_approved
                        ? "bg-emerald-100 text-emerald-700"
                        : "bg-amber-100 text-amber-700"
                    }`}>
                      {review.is_approved ? "অ্যাপ্রুভড" : "পেন্ডিং"}
                    </span>
                  </div>
                  <p className="text-xs text-muted-foreground mb-1">
                    📦 {review.product_name} • {new Date(review.created_at).toLocaleDateString("bn-BD")}
                  </p>
                  {review.comment && (
                    <p className="text-sm text-foreground/80 mt-1.5 bg-muted/40 p-2.5 rounded-lg">
                      "{review.comment}"
                    </p>
                  )}
                </div>
                <div className="flex items-center gap-1.5 shrink-0">
                  {!review.is_approved ? (
                    <Button size="sm" variant="default" className="gap-1 h-8" onClick={() => handleApprove(review.id, true)}>
                      <Check className="h-3.5 w-3.5" /> অ্যাপ্রুভ
                    </Button>
                  ) : (
                    <Button size="sm" variant="outline" className="gap-1 h-8" onClick={() => handleApprove(review.id, false)}>
                      <X className="h-3.5 w-3.5" /> রিজেক্ট
                    </Button>
                  )}
                  <Button size="sm" variant="destructive" className="gap-1 h-8" onClick={() => handleDelete(review.id)}>
                    <Trash2 className="h-3.5 w-3.5" />
                  </Button>
                </div>
              </div>
            </div>
          ))}
        </div>
      )}
    </div>
  );
};

export default AdminReviewsPanel;
