import { ReactNode } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { AlertCircle } from "lucide-react";
import { Button } from "@/components/ui/button";

// ── Confirmation Dialog ──
export const ConfirmDialog = ({ open, title, message, onConfirm, onCancel }: {
  open: boolean; title: string; message: string; onConfirm: () => void; onCancel: () => void;
}) => (
  <AnimatePresence>
    {open && (
      <motion.div initial={{ opacity: 0 }} animate={{ opacity: 1 }} exit={{ opacity: 0 }}
        className="fixed inset-0 z-[100] flex items-center justify-center bg-black/50 backdrop-blur-sm p-4">
        <motion.div initial={{ scale: 0.9, opacity: 0 }} animate={{ scale: 1, opacity: 1 }} exit={{ scale: 0.9, opacity: 0 }}
          className="bg-card rounded-2xl border border-border shadow-2xl p-6 max-w-sm w-full">
          <div className="flex items-center gap-3 mb-4">
            <div className="h-10 w-10 rounded-full bg-destructive/10 flex items-center justify-center">
              <AlertCircle className="h-5 w-5 text-destructive" />
            </div>
            <div>
              <h3 className="font-heading font-semibold">{title}</h3>
              <p className="text-sm text-muted-foreground">{message}</p>
            </div>
          </div>
          <div className="flex gap-2 justify-end">
            <Button variant="outline" size="sm" onClick={onCancel}>বাতিল</Button>
            <Button variant="destructive" size="sm" onClick={onConfirm}>ডিলিট করুন</Button>
          </div>
        </motion.div>
      </motion.div>
    )}
  </AnimatePresence>
);

// ── Stat Card ──
export const StatCard = ({ label, value, icon: Icon, trend, color }: {
  label: string; value: number; icon: any; trend?: string; color: string;
}) => (
  <motion.div initial={{ opacity: 0, y: 20 }} animate={{ opacity: 1, y: 0 }}
    className="bg-card rounded-2xl border border-border p-4 sm:p-5 hover:shadow-lg transition-shadow duration-300 group">
    <div className="flex items-start justify-between">
      <div>
        <p className="text-xs sm:text-sm text-muted-foreground mb-1">{label}</p>
        <p className="text-2xl sm:text-3xl font-bold font-heading">{value}</p>
        {trend && (
          <div className="flex items-center gap-1 mt-2">
            <span className="text-xs text-primary font-medium">{trend}</span>
          </div>
        )}
      </div>
      <div className={`h-10 w-10 sm:h-12 sm:w-12 rounded-xl ${color} flex items-center justify-center group-hover:scale-110 transition-transform duration-300`}>
        <Icon className="h-5 w-5 sm:h-6 sm:w-6" />
      </div>
    </div>
  </motion.div>
);

// ── Empty State ──
export const EmptyState = ({ icon: Icon, title, description, action }: {
  icon: any; title: string; description: string; action?: ReactNode;
}) => (
  <motion.div initial={{ opacity: 0 }} animate={{ opacity: 1 }}
    className="flex flex-col items-center justify-center py-12 sm:py-16 px-4">
    <div className="h-16 w-16 rounded-2xl bg-muted flex items-center justify-center mb-4">
      <Icon className="h-8 w-8 text-muted-foreground" />
    </div>
    <h3 className="font-heading font-semibold text-lg mb-1">{title}</h3>
    <p className="text-sm text-muted-foreground text-center max-w-sm mb-4">{description}</p>
    {action}
  </motion.div>
);

// ── Section Card ──
export const SectionCard = ({ title, description, icon: Icon, children, className = "" }: {
  title: string; description?: string; icon?: any; children: ReactNode; className?: string;
}) => (
  <div className={`bg-card rounded-2xl border border-border overflow-hidden ${className}`}>
    <div className="px-4 sm:px-6 py-3 sm:py-4 border-b border-border flex items-center gap-3">
      {Icon && (
        <div className="h-8 w-8 rounded-lg bg-primary/10 flex items-center justify-center flex-shrink-0">
          <Icon className="h-4 w-4 text-primary" />
        </div>
      )}
      <div>
        <h3 className="font-heading font-semibold text-sm">{title}</h3>
        {description && <p className="text-xs text-muted-foreground mt-0.5">{description}</p>}
      </div>
    </div>
    <div className="p-4 sm:p-6">{children}</div>
  </div>
);
