import { useState, useEffect } from "react";
import { supabase } from "@/integrations/supabase/client";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { useToast } from "@/hooks/use-toast";
import { SectionCard } from "./AdminShared";
import { Settings, MessageCircle, Phone, Mail, Globe, Save, Loader2, X } from "lucide-react";
import type { DBOrder } from "./adminTypes";

interface SmsTemplate {
  key: string;
  label: string;
  enabled: boolean;
  message: string;
  defaultMessage: string;
}

const defaultSmsTemplates: SmsTemplate[] = [
  { key: "processing", label: "Processing SMS", enabled: true, defaultMessage: "Dear {name} 💜\nThank you for shopping with\n{shop_name} ✨\n\nOrder: {items}\nTotal: {total}", message: "" },
  { key: "completed", label: "Completed SMS", enabled: true, defaultMessage: "💜 We appreciate your trust.\n\n— {shop_name}\n\nOrder #{order_id} completed!", message: "" },
  { key: "pending", label: "Pending Payment SMS", enabled: false, defaultMessage: "Dear {name}, Your order #{order_id} is Pending Payment.\nPlease complete your payment.", message: "" },
  { key: "cancelled", label: "Cancelled SMS", enabled: false, defaultMessage: "Dear {name}, Your order #{order_id} is Cancelled.", message: "" },
  { key: "failed", label: "Failed SMS", enabled: false, defaultMessage: "Dear {name}, Your order #{order_id} is Failed.", message: "" },
];

const AdminSmsPanel = ({ sendSms, orders }: { sendSms: (number: string, message: string) => Promise<any>; orders: DBOrder[] }) => {
  const [smsNumber, setSmsNumber] = useState("");
  const [smsMessage, setSmsMessage] = useState("");
  const [smsSending, setSmsSending] = useState(false);
  const [smsMode, setSmsMode] = useState<"single" | "bulk">("single");
  const [smsApiKey, setSmsApiKey] = useState("");
  const [smsTemplates, setSmsTemplates] = useState<SmsTemplate[]>(defaultSmsTemplates);
  const [savingTemplates, setSavingTemplates] = useState(false);
  const [testNumber, setTestNumber] = useState("");
  const [testSending, setTestSending] = useState(false);
  const { toast: smsToast } = useToast();
  const [showKeywords, setShowKeywords] = useState<string | null>(null);

  useEffect(() => {
    const loadSettings = async () => {
      const { data } = await supabase.from("site_settings").select("key, value");
      if (data) {
        const map: Record<string, string> = {};
        data.forEach(r => { if (r.value) map[r.key] = r.value; });
        if (map.sms_api_key) setSmsApiKey(map.sms_api_key);
        if (map.sms_templates) {
          try {
            const saved = JSON.parse(map.sms_templates) as Partial<SmsTemplate>[];
            setSmsTemplates(defaultSmsTemplates.map(dt => {
              const found = saved.find(s => s.key === dt.key);
              return found ? { ...dt, ...found, defaultMessage: dt.defaultMessage } : dt;
            }));
          } catch {}
        }
      }
    };
    loadSettings();
  }, []);

  const handleSaveTemplates = async () => {
    setSavingTemplates(true);
    try {
      const upsert = async (key: string, value: string) => {
        const { data: existing } = await supabase.from("site_settings").select("id").eq("key", key).maybeSingle();
        if (existing) await supabase.from("site_settings").update({ value, updated_at: new Date().toISOString() }).eq("key", key);
        else await supabase.from("site_settings").insert({ key, value });
      };
      if (smsApiKey) await upsert("sms_api_key", smsApiKey);
      const templateData = smsTemplates.map(t => ({ key: t.key, enabled: t.enabled, message: t.message }));
      await upsert("sms_templates", JSON.stringify(templateData));
      smsToast({ title: "✅ SMS সেটিংস সেভ হয়েছে!" });
    } catch {
      smsToast({ title: "সেভ করতে সমস্যা হয়েছে", variant: "destructive" });
    } finally { setSavingTemplates(false); }
  };

  const handleSendSms = async () => {
    if (!smsNumber || !smsMessage) { smsToast({ title: "নম্বর ও মেসেজ দিন", variant: "destructive" }); return; }
    setSmsSending(true);
    try { await sendSms(smsNumber, smsMessage); smsToast({ title: "✅ SMS পাঠানো হয়েছে!" }); setSmsNumber(""); setSmsMessage(""); }
    catch { smsToast({ title: "SMS পাঠাতে সমস্যা হয়েছে", variant: "destructive" }); }
    finally { setSmsSending(false); }
  };

  const handleBulkSms = async () => {
    if (!smsMessage) { smsToast({ title: "মেসেজ দিন", variant: "destructive" }); return; }
    const uniquePhones = [...new Set(orders.filter(o => o.customer_phone).map(o => o.customer_phone!))];
    if (uniquePhones.length === 0) { smsToast({ title: "কোনো কাস্টমারের ফোন নম্বর নেই", variant: "destructive" }); return; }
    setSmsSending(true);
    try { for (const phone of uniquePhones) { await sendSms(phone, smsMessage); } smsToast({ title: `✅ ${uniquePhones.length}টি SMS পাঠানো হয়েছে!` }); setSmsMessage(""); }
    catch { smsToast({ title: "কিছু SMS পাঠাতে সমস্যা হয়েছে", variant: "destructive" }); }
    finally { setSmsSending(false); }
  };

  const handleTestSms = async () => {
    if (!testNumber) { smsToast({ title: "টেস্ট নম্বর দিন", variant: "destructive" }); return; }
    setTestSending(true);
    try { await sendSms(testNumber, "This is a test SMS from your shop."); smsToast({ title: "✅ টেস্ট SMS পাঠানো হয়েছে!" }); }
    catch { smsToast({ title: "টেস্ট SMS পাঠাতে সমস্যা হয়েছে", variant: "destructive" }); }
    finally { setTestSending(false); }
  };

  const updateTemplate = (key: string, field: Partial<SmsTemplate>) => {
    setSmsTemplates(prev => prev.map(t => t.key === key ? { ...t, ...field } : t));
  };

  const keywordGroups = [
    { label: "Shop", tags: ["{shop_name}", "{shop_email}", "{shop_url}", "{shop_phone}"] },
    { label: "Order", tags: ["{order_id}", "{order_status}", "{order_amount}", "{order_product}", "{order_product_with_qty}", "{order_currency}"] },
    { label: "Customer", tags: ["{name}", "{phone}", "{email}"] },
  ];

  const insertTag = (templateKey: string, tag: string) => {
    const t = smsTemplates.find(t => t.key === templateKey);
    if (!t) return;
    const current = t.message || t.defaultMessage;
    updateTemplate(templateKey, { message: current + tag });
  };

  return (
    <div className="space-y-6 max-w-3xl">
      <SectionCard title="SMS API Key" description="Hakvolution SMS API Key সেট করুন" icon={Settings}>
        <div className="space-y-1.5">
          <Label className="text-xs">API Key</Label>
          <Input value={smsApiKey} onChange={e => setSmsApiKey(e.target.value)} placeholder="আপনার SMS API Key" className="rounded-xl font-mono text-sm" />
        </div>
      </SectionCard>

      <SectionCard title="Order SMS Templates" description="অর্ডার স্ট্যাটাস অনুযায়ী অটো SMS কনফিগার করুন" icon={MessageCircle}>
        <div className="space-y-5">
          {smsTemplates.map((template) => (
            <div key={template.key} className="border border-border rounded-xl p-3 sm:p-4 space-y-3">
              <div className="flex items-center justify-between">
                <h4 className="font-semibold text-sm">{template.label}</h4>
                <div className="flex items-center gap-2">
                  <span className="text-xs text-muted-foreground hidden sm:inline">Enable</span>
                  <Switch checked={template.enabled} onCheckedChange={(checked) => updateTemplate(template.key, { enabled: checked })} />
                </div>
              </div>
              <textarea
                value={template.message || template.defaultMessage}
                onChange={e => updateTemplate(template.key, { message: e.target.value })}
                className="flex w-full rounded-xl border border-input bg-background px-3 py-2 text-sm ring-offset-background placeholder:text-muted-foreground focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2 min-h-[100px] resize-y"
              />
              <div className="flex items-center gap-2 flex-wrap">
                <button type="button" onClick={() => setShowKeywords(showKeywords === template.key ? null : template.key)}
                  className="text-xs font-medium text-primary border border-primary/30 bg-primary/5 px-2.5 py-1 rounded-lg hover:bg-primary/10 transition-colors">
                  Keywords
                </button>
              </div>
              {showKeywords === template.key && (
                <div className="border border-border rounded-xl p-3 sm:p-4 space-y-3 bg-muted/30 relative">
                  <button type="button" onClick={() => setShowKeywords(null)} className="absolute top-2 right-2 h-6 w-6 rounded-full bg-muted flex items-center justify-center hover:bg-muted-foreground/20 transition-colors">
                    <X className="h-3.5 w-3.5" />
                  </button>
                  {keywordGroups.map(group => (
                    <div key={group.label}>
                      <p className="text-xs font-semibold mb-1.5">{group.label}</p>
                      <div className="flex flex-wrap gap-1.5">
                        {group.tags.map(tag => (
                          <button key={tag} type="button" onClick={() => insertTag(template.key, tag)}
                            className="text-[11px] font-mono text-primary bg-primary/10 border border-primary/20 px-2 py-1 rounded-md hover:bg-primary/20 transition-colors cursor-pointer">
                            {tag}
                          </button>
                        ))}
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </div>
          ))}
          <Button onClick={handleSaveTemplates} disabled={savingTemplates} className="gap-1.5 rounded-xl w-full sm:w-auto">
            {savingTemplates ? <Loader2 className="h-4 w-4 animate-spin" /> : <Save className="h-4 w-4" />}
            {savingTemplates ? "সেভ হচ্ছে..." : "Save Changes"}
          </Button>
        </div>
      </SectionCard>

      <SectionCard title="Send Test SMS" icon={Phone}>
        <div className="flex flex-col sm:flex-row gap-2 sm:items-end">
          <div className="flex-1 space-y-1.5">
            <Input value={testNumber} onChange={e => setTestNumber(e.target.value)} placeholder="017XXXXXXXX" className="rounded-xl" />
          </div>
          <Button onClick={handleTestSms} disabled={testSending} className="gap-1.5 rounded-xl w-full sm:w-auto">
            {testSending ? <Loader2 className="h-4 w-4 animate-spin" /> : <MessageCircle className="h-4 w-4" />}
            Send Test SMS
          </Button>
        </div>
      </SectionCard>

      <SectionCard title="SMS পাঠান" icon={MessageCircle}>
        <div className="flex gap-2 mb-6">
          <button onClick={() => setSmsMode("single")}
            className={`px-3 sm:px-4 py-2 rounded-xl text-sm font-medium transition-all ${smsMode === "single" ? "bg-primary text-primary-foreground" : "bg-muted text-muted-foreground"}`}>
            একজনকে পাঠান
          </button>
          <button onClick={() => setSmsMode("bulk")}
            className={`px-3 sm:px-4 py-2 rounded-xl text-sm font-medium transition-all ${smsMode === "bulk" ? "bg-primary text-primary-foreground" : "bg-muted text-muted-foreground"}`}>
            সবাইকে পাঠান
          </button>
        </div>
        <div className="space-y-4">
          {smsMode === "single" && (
            <div className="space-y-1.5">
              <Label className="text-xs flex items-center gap-1.5"><Phone className="h-3 w-3" />ফোন নম্বর</Label>
              <Input value={smsNumber} onChange={e => setSmsNumber(e.target.value)} placeholder="01XXXXXXXXX" className="rounded-xl" />
            </div>
          )}
          {smsMode === "bulk" && (
            <div className="rounded-xl bg-muted/50 border border-border p-4">
              <p className="text-sm text-muted-foreground">
                <span className="font-semibold text-foreground">{[...new Set(orders.filter(o => o.customer_phone).map(o => o.customer_phone!))].length}</span> জন ইউনিক কাস্টমারকে SMS পাঠানো হবে
              </p>
            </div>
          )}
          <div className="space-y-1.5">
            <Label className="text-xs flex items-center gap-1.5"><Mail className="h-3 w-3" />মেসেজ</Label>
            <textarea value={smsMessage} onChange={e => setSmsMessage(e.target.value)} placeholder="আপনার মেসেজ লিখুন..."
              className="flex w-full rounded-xl border border-input bg-background px-3 py-2 text-sm ring-offset-background placeholder:text-muted-foreground focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2 min-h-[120px] resize-y" />
          </div>
          <Button onClick={smsMode === "single" ? handleSendSms : handleBulkSms} disabled={smsSending} className="gap-1.5 rounded-xl w-full sm:w-auto">
            {smsSending ? <Loader2 className="h-4 w-4 animate-spin" /> : <MessageCircle className="h-4 w-4" />}
            {smsSending ? "পাঠানো হচ্ছে..." : smsMode === "single" ? "SMS পাঠান" : "সবাইকে SMS পাঠান"}
          </Button>
        </div>
      </SectionCard>

      <SectionCard title="SMS টেমপ্লেট" icon={Globe}>
        <div className="space-y-3">
          {[
            { label: "অর্ডার কনফার্মেশন", msg: "প্রিয় গ্রাহক, আপনার অর্ডার সফলভাবে গ্রহণ করা হয়েছে। শীঘ্রই প্রসেস করা হবে। ধন্যবাদ!" },
            { label: "অর্ডার সম্পন্ন", msg: "প্রিয় গ্রাহক, আপনার অর্ডার সফলভাবে সম্পন্ন হয়েছে। পুনরায় কেনাকাটার আমন্ত্রণ রইলো। ধন্যবাদ!" },
            { label: "নতুন অফার", msg: "🎉 বিশেষ অফার! সীমিত সময়ের জন্য সকল প্রোডাক্টে বিশেষ ছাড়। এখনই অর্ডার করুন!" },
            { label: "পেমেন্ট রিমাইন্ডার", msg: "প্রিয় গ্রাহক, আপনার অর্ডারের পেমেন্ট এখনো সম্পন্ন হয়নি। দ্রুত পেমেন্ট সম্পন্ন করুন।" },
          ].map((t, i) => (
            <button key={i} onClick={() => setSmsMessage(t.msg)}
              className="w-full text-left p-3 rounded-xl border border-border hover:bg-muted/50 transition-colors">
              <p className="text-sm font-medium text-foreground">{t.label}</p>
              <p className="text-xs text-muted-foreground mt-0.5 line-clamp-1">{t.msg}</p>
            </button>
          ))}
        </div>
      </SectionCard>
    </div>
  );
};

export default AdminSmsPanel;
