import { useState, useEffect } from "react";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Trash2, Search, Download, Mail, Loader2, Send, Eye, X } from "lucide-react";
import { ConfirmDialog, EmptyState, SectionCard } from "@/components/admin/AdminShared";
import { motion, AnimatePresence } from "framer-motion";

interface Subscriber {
  id: string;
  email: string;
  created_at: string;
}

const AdminSubscribersPanel = () => {
  const [subscribers, setSubscribers] = useState<Subscriber[]>([]);
  const [loading, setLoading] = useState(true);
  const [shopName, setShopName] = useState("Digital Shop");
  const [search, setSearch] = useState("");
  const [deleteId, setDeleteId] = useState<string | null>(null);
  const [showCompose, setShowCompose] = useState(false);
  const [subject, setSubject] = useState("");
  const [body, setBody] = useState("");
  const [sending, setSending] = useState(false);
  const [sendProgress, setSendProgress] = useState({ sent: 0, total: 0 });
  const [showPreview, setShowPreview] = useState(false);
  const { toast } = useToast();

  const fetchSubscribers = async () => {
    setLoading(true);
    const { data, error } = await supabase
      .from("subscribers")
      .select("*")
      .order("created_at", { ascending: false });
    if (!error && data) setSubscribers(data);
    setLoading(false);
  };

  useEffect(() => {
    fetchSubscribers();
    supabase.from("site_settings").select("value").eq("key", "site_title").maybeSingle().then(({ data }) => {
      if (data?.value) setShopName(data.value);
    });
  }, []);

  const handleDelete = async () => {
    if (!deleteId) return;
    const { error } = await supabase.from("subscribers").delete().eq("id", deleteId);
    if (!error) {
      setSubscribers(s => s.filter(sub => sub.id !== deleteId));
      toast({ title: "সাবস্ক্রাইবার ডিলিট হয়েছে" });
    } else {
      toast({ title: "ডিলিট করতে সমস্যা হয়েছে", variant: "destructive" });
    }
    setDeleteId(null);
  };

  const handleExportCSV = () => {
    const csv = "Email,Subscribed At\n" + filtered.map(s => `${s.email},${s.created_at}`).join("\n");
    const blob = new Blob([csv], { type: "text/csv" });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = `subscribers-${new Date().toISOString().slice(0, 10)}.csv`;
    a.click();
    URL.revokeObjectURL(url);
  };

  const buildNewsletterHtml = (contentBody: string, shopName: string) => {
    const lines = contentBody.split("\n").filter(l => l.trim()).map(l => `<p style="margin:0 0 12px;color:#374151;font-size:15px;line-height:1.7">${l}</p>`).join("");
    return `<div style="font-family:'Segoe UI',Arial,sans-serif;max-width:600px;margin:0 auto;background:#f7f7f7;padding:20px 0">
      <div style="background:#fff;border-radius:8px;overflow:hidden;border:1px solid #e0e0e0;margin:0 20px">
        <div style="background:linear-gradient(135deg,#6366f1,#8b5cf6);padding:28px 30px;text-align:center">
          <h1 style="color:#fff;margin:0;font-size:22px;font-weight:700">📬 নিউজলেটার</h1>
          <p style="color:rgba(255,255,255,.85);margin:8px 0 0;font-size:14px">${shopName}</p>
        </div>
        <div style="padding:30px">${lines}</div>
        <div style="border-top:1px solid #e5e7eb;padding:20px 30px;text-align:center;background:#fafafa">
          <p style="color:#999;font-size:12px;margin:0">© ${new Date().getFullYear()} ${shopName}</p>
          <p style="color:#bbb;font-size:11px;margin:8px 0 0">আপনি এই ইমেইল পাচ্ছেন কারণ আপনি আমাদের নিউজলেটারে সাবস্ক্রাইব করেছেন।</p>
        </div>
      </div>
    </div>`;
  };

  const handleSendNewsletter = async () => {
    if (!subject.trim() || !body.trim()) {
      toast({ title: "সাবজেক্ট ও বডি দিন", variant: "destructive" });
      return;
    }
    if (subscribers.length === 0) {
      toast({ title: "কোনো সাবস্ক্রাইবার নেই", variant: "destructive" });
      return;
    }

    setSending(true);
    const html = buildNewsletterHtml(body, shopName);
    const total = subscribers.length;
    setSendProgress({ sent: 0, total });
    let successCount = 0;
    let failCount = 0;

    for (const sub of subscribers) {
      try {
        await supabase.functions.invoke("send-email", {
          body: { to: sub.email, subject: subject.trim(), html, text: body.trim() },
        });
        successCount++;
      } catch {
        failCount++;
      }
      setSendProgress({ sent: successCount + failCount, total });
    }

    setSending(false);
    toast({
      title: `✅ ${successCount}/${total} জনকে ইমেইল পাঠানো হয়েছে`,
      description: failCount > 0 ? `${failCount}টি ব্যর্থ হয়েছে` : undefined,
      variant: failCount > 0 ? "destructive" : undefined,
    });
    if (failCount === 0) {
      setSubject("");
      setBody("");
      setShowCompose(false);
    }
  };

  const filtered = subscribers.filter(s => s.email.toLowerCase().includes(search.toLowerCase()));

  return (
    <div className="space-y-4">
      {/* Header */}
      <div className="flex flex-col sm:flex-row items-start sm:items-center justify-between gap-3">
        <p className="text-sm text-muted-foreground">মোট {subscribers.length} জন সাবস্ক্রাইবার</p>
        <div className="flex items-center gap-2 w-full sm:w-auto">
          <div className="relative flex-1 sm:flex-initial">
            <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-3.5 w-3.5 text-muted-foreground" />
            <Input
              placeholder="ইমেইল খুঁজুন..."
              value={search}
              onChange={e => setSearch(e.target.value)}
              className="pl-9 h-9 rounded-xl text-sm w-full sm:w-56"
            />
          </div>
          <Button size="sm" variant="outline" className="gap-1.5 rounded-xl" onClick={handleExportCSV} disabled={filtered.length === 0}>
            <Download className="h-3.5 w-3.5" /> CSV
          </Button>
          <Button size="sm" className="gap-1.5 rounded-xl" onClick={() => setShowCompose(!showCompose)} disabled={subscribers.length === 0}>
            {showCompose ? <X className="h-3.5 w-3.5" /> : <Send className="h-3.5 w-3.5" />}
            {showCompose ? "বাতিল" : "নিউজলেটার"}
          </Button>
        </div>
      </div>

      {/* Newsletter Compose */}
      <AnimatePresence>
        {showCompose && (
          <motion.div initial={{ opacity: 0, height: 0 }} animate={{ opacity: 1, height: "auto" }} exit={{ opacity: 0, height: 0 }}>
            <SectionCard title="📬 নিউজলেটার পাঠান" icon={Send}>
              <div className="space-y-4">
                <div className="space-y-1.5">
                  <Label className="text-xs">সাবজেক্ট *</Label>
                  <Input
                    placeholder="ইমেইলের সাবজেক্ট লিখুন..."
                    value={subject}
                    onChange={e => setSubject(e.target.value)}
                    className="rounded-xl"
                  />
                </div>
                <div className="space-y-1.5">
                  <Label className="text-xs">বডি * (প্রতিটি লাইন আলাদা প্যারাগ্রাফ হবে)</Label>
                  <Textarea
                    placeholder="নিউজলেটারের বিষয়বস্তু লিখুন..."
                    value={body}
                    onChange={e => setBody(e.target.value)}
                    className="rounded-xl min-h-[150px]"
                  />
                </div>
                <div className="flex flex-wrap items-center gap-2">
                  <Button size="sm" variant="outline" className="gap-1.5 rounded-xl" onClick={() => setShowPreview(true)} disabled={!body.trim()}>
                    <Eye className="h-3.5 w-3.5" /> প্রিভিউ
                  </Button>
                  <Button size="sm" className="gap-1.5 rounded-xl" onClick={handleSendNewsletter} disabled={sending || !subject.trim() || !body.trim()}>
                    {sending ? <Loader2 className="h-3.5 w-3.5 animate-spin" /> : <Send className="h-3.5 w-3.5" />}
                    {sending ? `পাঠানো হচ্ছে (${sendProgress.sent}/${sendProgress.total})` : `${subscribers.length} জনকে পাঠান`}
                  </Button>
                </div>
              </div>
            </SectionCard>
          </motion.div>
        )}
      </AnimatePresence>

      {/* Preview Modal */}
      <AnimatePresence>
        {showPreview && (
          <motion.div
            initial={{ opacity: 0 }} animate={{ opacity: 1 }} exit={{ opacity: 0 }}
            className="fixed inset-0 z-[100] flex items-center justify-center bg-black/50 backdrop-blur-sm p-4"
            onClick={() => setShowPreview(false)}
          >
            <motion.div
              initial={{ scale: 0.9, opacity: 0 }} animate={{ scale: 1, opacity: 1 }} exit={{ scale: 0.9, opacity: 0 }}
              className="bg-card rounded-2xl border shadow-2xl max-w-2xl w-full max-h-[80vh] overflow-hidden flex flex-col"
              onClick={e => e.stopPropagation()}
            >
              <div className="px-6 py-4 border-b flex items-center justify-between">
                <h3 className="font-heading font-semibold text-sm">ইমেইল প্রিভিউ</h3>
                <Button size="sm" variant="ghost" className="h-7 w-7 p-0" onClick={() => setShowPreview(false)}>
                  <X className="h-4 w-4" />
                </Button>
              </div>
              <div className="p-4 overflow-y-auto flex-1 bg-muted/30">
                <div className="text-xs text-muted-foreground mb-2">Subject: <strong>{subject || "(খালি)"}</strong></div>
                <div
                  className="bg-background rounded-xl border overflow-hidden"
                  dangerouslySetInnerHTML={{ __html: buildNewsletterHtml(body, shopName) }}
                />
              </div>
            </motion.div>
          </motion.div>
        )}
      </AnimatePresence>

      {/* Subscribers Table */}
      {loading ? (
        <div className="flex justify-center py-12"><Loader2 className="h-6 w-6 animate-spin text-muted-foreground" /></div>
      ) : (
        <div className="bg-card rounded-2xl border overflow-hidden">
          <div className="overflow-x-auto">
            <table className="w-full text-sm">
              <thead>
                <tr className="border-b bg-muted/30">
                  <th className="px-4 py-3 text-left font-medium text-muted-foreground">#</th>
                  <th className="px-4 py-3 text-left font-medium text-muted-foreground">ইমেইল</th>
                  <th className="px-4 py-3 text-left font-medium text-muted-foreground">তারিখ</th>
                  <th className="px-4 py-3 text-right font-medium text-muted-foreground">অ্যাকশন</th>
                </tr>
              </thead>
              <tbody>
                {filtered.map((sub, i) => (
                  <motion.tr
                    key={sub.id}
                    initial={{ opacity: 0 }}
                    animate={{ opacity: 1 }}
                    className="border-b last:border-0 hover:bg-muted/20 transition-colors"
                  >
                    <td className="px-4 py-3 text-muted-foreground">{i + 1}</td>
                    <td className="px-4 py-3 font-medium">{sub.email}</td>
                    <td className="px-4 py-3 text-muted-foreground">
                      {new Date(sub.created_at).toLocaleDateString("bn-BD")}
                    </td>
                    <td className="px-4 py-3 text-right">
                      <Button size="sm" variant="ghost" className="h-7 w-7 p-0 text-destructive hover:text-destructive" onClick={() => setDeleteId(sub.id)}>
                        <Trash2 className="h-3.5 w-3.5" />
                      </Button>
                    </td>
                  </motion.tr>
                ))}
                {filtered.length === 0 && (
                  <tr><td colSpan={4}>
                    <EmptyState icon={Mail} title="কোনো সাবস্ক্রাইবার নেই" description="এখনো কেউ সাবস্ক্রাইব করেনি।" />
                  </td></tr>
                )}
              </tbody>
            </table>
          </div>
        </div>
      )}

      <ConfirmDialog
        open={!!deleteId}
        title="সাবস্ক্রাইবার ডিলিট"
        message="এই সাবস্ক্রাইবার ডিলিট করতে চান?"
        onConfirm={handleDelete}
        onCancel={() => setDeleteId(null)}
      />
    </div>
  );
};

export default AdminSubscribersPanel;
