import { useState, useEffect } from "react";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import {
  Table, TableBody, TableCell, TableHead, TableHeader, TableRow
} from "@/components/ui/table";
import { SectionCard, EmptyState } from "@/components/admin/AdminShared";
import {
  Search, Users, Edit2, Save, XCircle, Loader2, Phone, Shield,
  Calendar, ChevronDown, ChevronUp, User, KeyRound, ShieldCheck, ShieldOff
} from "lucide-react";
import { motion, AnimatePresence } from "framer-motion";

type UserRole = "admin" | "reseller" | "user";

interface UserProfile {
  id: string;
  user_id: string;
  name: string | null;
  email: string | null;
  phone: string | null;
  avatar_url: string | null;
  created_at: string;
  role: UserRole;
  order_count: number;
}

const AdminUsersPanel = () => {
  const [users, setUsers] = useState<UserProfile[]>([]);
  const [loading, setLoading] = useState(true);
  const [search, setSearch] = useState("");
  const [editingUser, setEditingUser] = useState<string | null>(null);
  const [editFields, setEditFields] = useState({ name: "", email: "", phone: "" });
  const [saving, setSaving] = useState(false);
  const [expandedUser, setExpandedUser] = useState<string | null>(null);
  const [passwordModal, setPasswordModal] = useState<string | null>(null);
  const [newPassword, setNewPassword] = useState("");
  const [changingPassword, setChangingPassword] = useState(false);
  const [togglingRole, setTogglingRole] = useState<string | null>(null);
  const { toast } = useToast();

  const fetchUsers = async () => {
    setLoading(true);
    try {
      // Fetch profiles
      const { data: profiles, error } = await supabase
        .from("profiles")
        .select("*")
        .order("created_at", { ascending: false });

      if (error) throw error;

      // Fetch admin roles
      const { data: roles } = await supabase
        .from("user_roles")
        .select("user_id, role");

      // Fetch order counts per user
      const { data: orders } = await supabase
        .from("orders")
        .select("user_id");

      const orderCounts: Record<string, number> = {};
      orders?.forEach(o => {
        if (o.user_id) {
          orderCounts[o.user_id] = (orderCounts[o.user_id] || 0) + 1;
        }
      });

      const adminSet = new Set(
        roles?.filter(r => r.role === "admin").map(r => r.user_id) || []
      );
      const resellerSet = new Set(
        roles?.filter(r => r.role === "reseller").map(r => r.user_id) || []
      );

      const mapped: UserProfile[] = (profiles || []).map(p => ({
        id: p.id,
        user_id: p.user_id,
        name: p.name,
        email: p.email,
        phone: p.phone,
        avatar_url: p.avatar_url,
        created_at: p.created_at,
        role: adminSet.has(p.user_id) ? "admin" as UserRole : resellerSet.has(p.user_id) ? "reseller" as UserRole : "user" as UserRole,
        order_count: orderCounts[p.user_id] || 0,
      }));

      setUsers(mapped);
    } catch (err: any) {
      toast({ title: "ইউজার লোড করতে সমস্যা", description: err.message, variant: "destructive" });
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => { fetchUsers(); }, []);

  const filtered = users.filter(u => {
    const q = search.toLowerCase();
    return (
      (u.name || "").toLowerCase().includes(q) ||
      (u.email || "").toLowerCase().includes(q) ||
      (u.phone || "").toLowerCase().includes(q)
    );
  });

  const startEdit = (u: UserProfile) => {
    setEditingUser(u.user_id);
    setEditFields({ name: u.name || "", email: u.email || "", phone: u.phone || "" });
  };

  const cancelEdit = () => {
    setEditingUser(null);
    setEditFields({ name: "", email: "", phone: "" });
  };

  const handleSave = async (userId: string) => {
    setSaving(true);
    try {
      const { error } = await supabase
        .from("profiles")
        .update({
          name: editFields.name || null,
          email: editFields.email || null,
          phone: editFields.phone || null,
        })
        .eq("user_id", userId);

      if (error) throw error;

      toast({ title: "✅ ইউজার আপডেট হয়েছে" });
      setEditingUser(null);
      fetchUsers();
    } catch (err: any) {
      toast({ title: "আপডেট ব্যর্থ", description: err.message, variant: "destructive" });
    } finally {
      setSaving(false);
    }
  };

  const toggleExpand = (userId: string) => {
    setExpandedUser(prev => (prev === userId ? null : userId));
  };

  // New: Change password
  const handleChangePassword = async () => {
    if (!passwordModal || newPassword.length < 6) {
      toast({ title: "পাসওয়ার্ড কমপক্ষে ৬ অক্ষর হতে হবে", variant: "destructive" });
      return;
    }
    setChangingPassword(true);
    try {
      const { data: { session } } = await supabase.auth.getSession();
      const res = await supabase.functions.invoke("admin-user-actions", {
        body: { action: "change_password", user_id: passwordModal, password: newPassword },
      });
      if (res.error || res.data?.error) {
        throw new Error(res.data?.error || res.error?.message || "Failed");
      }
      toast({ title: "✅ পাসওয়ার্ড পরিবর্তন হয়েছে" });
      setPasswordModal(null);
      setNewPassword("");
    } catch (err: any) {
      toast({ title: "পাসওয়ার্ড পরিবর্তন ব্যর্থ", description: err.message, variant: "destructive" });
    } finally {
      setChangingPassword(false);
    }
  };

  // New: Set role
  const handleSetRole = async (userId: string, newRole: UserRole) => {
    setTogglingRole(userId);
    try {
      const res = await supabase.functions.invoke("admin-user-actions", {
        body: { action: "set_role", user_id: userId, role: newRole },
      });
      if (res.error || res.data?.error) {
        throw new Error(res.data?.error || res.error?.message || "Failed");
      }
      const roleLabels: Record<UserRole, string> = { admin: "🛡️ অ্যাডমিন", reseller: "🏪 রিসেলার", user: "👤 ইউজার" };
      toast({ title: `${roleLabels[newRole]} রোল সেট করা হয়েছে` });
      fetchUsers();
    } catch (err: any) {
      toast({ title: "রোল পরিবর্তন ব্যর্থ", description: err.message, variant: "destructive" });
    } finally {
      setTogglingRole(null);
    }
  };

  const RoleBadge = ({ role }: { role: UserRole }) => {
    if (role === "admin") return (
      <Badge className="bg-primary/10 text-primary border-0 text-xs gap-1"><Shield className="h-3 w-3" /> Admin</Badge>
    );
    if (role === "reseller") return (
      <Badge className="bg-accent text-accent-foreground border-0 text-xs gap-1">🏪 Reseller</Badge>
    );
    return <Badge variant="outline" className="text-xs">User</Badge>;
  };

  // Role selector buttons
  const RoleSelector = ({ u }: { u: UserProfile }) => (
    <div className="flex gap-1 flex-wrap">
      {(["user", "reseller", "admin"] as UserRole[]).map(r => (
        <Button
          key={r}
          size="sm"
          variant={u.role === r ? "default" : "outline"}
          className="text-[11px] h-7 px-2"
          disabled={togglingRole === u.user_id || u.role === r}
          onClick={() => handleSetRole(u.user_id, r)}
        >
          {togglingRole === u.user_id ? <Loader2 className="h-3 w-3 animate-spin" /> : 
           r === "admin" ? "🛡️ Admin" : r === "reseller" ? "🏪 Reseller" : "👤 User"}
        </Button>
      ))}
    </div>
  );

  // Action buttons for a user (shared between desktop & mobile)
  const UserActions = ({ u, vertical = false }: { u: UserProfile; vertical?: boolean }) => (
    <div className={`flex ${vertical ? "flex-col" : ""} gap-1.5 ${vertical ? "" : "justify-end"}`}>
      <Button size="sm" variant="outline" className="gap-1.5 text-xs" onClick={() => startEdit(u)}>
        <Edit2 className="h-3.5 w-3.5" /> এডিট
      </Button>
      <Button size="sm" variant="outline" className="gap-1.5 text-xs" onClick={() => { setPasswordModal(u.user_id); setNewPassword(""); }}>
        <KeyRound className="h-3.5 w-3.5" /> পাসওয়ার্ড
      </Button>
    </div>
  );

  if (loading) {
    return (
      <div className="flex items-center justify-center py-20">
        <Loader2 className="h-8 w-8 animate-spin text-primary" />
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Password Change Modal */}
      <AnimatePresence>
        {passwordModal && (
          <motion.div initial={{ opacity: 0 }} animate={{ opacity: 1 }} exit={{ opacity: 0 }}
            className="fixed inset-0 z-[100] flex items-center justify-center bg-black/50 backdrop-blur-sm p-4"
            onClick={() => { setPasswordModal(null); setNewPassword(""); }}
          >
            <motion.div initial={{ scale: 0.9, opacity: 0 }} animate={{ scale: 1, opacity: 1 }} exit={{ scale: 0.9, opacity: 0 }}
              className="bg-card rounded-2xl border border-border shadow-2xl p-6 max-w-sm w-full"
              onClick={e => e.stopPropagation()}
            >
              <div className="flex items-center gap-3 mb-4">
                <div className="h-10 w-10 rounded-full bg-primary/10 flex items-center justify-center">
                  <KeyRound className="h-5 w-5 text-primary" />
                </div>
                <div>
                  <h3 className="font-heading font-semibold">পাসওয়ার্ড পরিবর্তন</h3>
                  <p className="text-xs text-muted-foreground">
                    {users.find(u => u.user_id === passwordModal)?.email || "ইউজার"}
                  </p>
                </div>
              </div>
              <div className="space-y-3">
                <div>
                  <Label className="text-xs">নতুন পাসওয়ার্ড</Label>
                  <Input
                    type="password"
                    value={newPassword}
                    onChange={e => setNewPassword(e.target.value)}
                    placeholder="কমপক্ষে ৬ অক্ষর"
                    className="rounded-xl"
                    onKeyDown={e => { if (e.key === "Enter") handleChangePassword(); }}
                  />
                </div>
                <div className="flex gap-2">
                  <Button variant="outline" className="flex-1" onClick={() => { setPasswordModal(null); setNewPassword(""); }}>
                    বাতিল
                  </Button>
                  <Button className="flex-1 gap-1.5" onClick={handleChangePassword} disabled={changingPassword || newPassword.length < 6}>
                    {changingPassword ? <Loader2 className="h-4 w-4 animate-spin" /> : <Save className="h-4 w-4" />}
                    পরিবর্তন করুন
                  </Button>
                </div>
              </div>
            </motion.div>
          </motion.div>
        )}
      </AnimatePresence>

      {/* Stats */}
      <div className="grid grid-cols-2 sm:grid-cols-4 gap-4">
        <div className="bg-card rounded-2xl border border-border p-4">
          <p className="text-xs text-muted-foreground">মোট ইউজার</p>
          <p className="text-2xl font-bold font-heading mt-1">{users.length}</p>
        </div>
        <div className="bg-card rounded-2xl border border-border p-4">
          <p className="text-xs text-muted-foreground">অ্যাডমিন</p>
          <p className="text-2xl font-bold font-heading mt-1">{users.filter(u => u.role === "admin").length}</p>
        </div>
        <div className="bg-card rounded-2xl border border-border p-4">
          <p className="text-xs text-muted-foreground">রিসেলার</p>
          <p className="text-2xl font-bold font-heading mt-1">{users.filter(u => u.role === "reseller").length}</p>
        </div>
        <div className="bg-card rounded-2xl border border-border p-4">
          <p className="text-xs text-muted-foreground">অর্ডারকৃত ইউজার</p>
          <p className="text-2xl font-bold font-heading mt-1">{users.filter(u => u.order_count > 0).length}</p>
        </div>
      </div>

      {/* Search */}
      <div className="relative">
        <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
        <Input
          placeholder="নাম, ইমেইল বা ফোন দিয়ে খুঁজুন..."
          value={search}
          onChange={e => setSearch(e.target.value)}
          className="pl-10 rounded-xl"
        />
      </div>

      {/* User List */}
      {filtered.length === 0 ? (
        <EmptyState icon={Users} title="কোন ইউজার পাওয়া যায়নি" description="সার্চ পরিবর্তন করে আবার চেষ্টা করুন" />
      ) : (
        <SectionCard title={`ইউজার তালিকা (${filtered.length})`} icon={Users}>
          {/* Desktop Table */}
          <div className="hidden md:block overflow-x-auto">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>ইউজার</TableHead>
                  <TableHead>ইমেইল</TableHead>
                  <TableHead>ফোন</TableHead>
                  <TableHead>অর্ডার</TableHead>
                  <TableHead>রোল</TableHead>
                  <TableHead>যোগদান</TableHead>
                  <TableHead className="text-right">অ্যাকশন</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {filtered.map(u => (
                  <TableRow key={u.user_id}>
                    <TableCell>
                      <div className="flex items-center gap-3">
                        {u.avatar_url ? (
                          <img src={u.avatar_url} alt="" className="h-8 w-8 rounded-full object-cover" />
                        ) : (
                          <div className="h-8 w-8 rounded-full bg-primary/10 flex items-center justify-center">
                            <User className="h-4 w-4 text-primary" />
                          </div>
                        )}
                        {editingUser === u.user_id ? (
                          <Input value={editFields.name} onChange={e => setEditFields(f => ({ ...f, name: e.target.value }))}
                            className="h-8 text-sm w-32" placeholder="নাম" />
                        ) : (
                          <span className="font-medium text-sm">{u.name || "—"}</span>
                        )}
                      </div>
                    </TableCell>
                    <TableCell>
                      {editingUser === u.user_id ? (
                        <Input value={editFields.email} onChange={e => setEditFields(f => ({ ...f, email: e.target.value }))}
                          className="h-8 text-sm w-40" placeholder="ইমেইল" />
                      ) : (
                        <span className="text-sm text-muted-foreground">{u.email || "—"}</span>
                      )}
                    </TableCell>
                    <TableCell>
                      {editingUser === u.user_id ? (
                        <Input value={editFields.phone} onChange={e => setEditFields(f => ({ ...f, phone: e.target.value }))}
                          className="h-8 text-sm w-32" placeholder="ফোন" />
                      ) : (
                        <span className="text-sm text-muted-foreground">{u.phone || "—"}</span>
                      )}
                    </TableCell>
                    <TableCell>
                      <Badge variant="secondary" className="text-xs">{u.order_count}</Badge>
                    </TableCell>
                    <TableCell>
                      <div className="space-y-1">
                        <RoleBadge role={u.role} />
                        <RoleSelector u={u} />
                      </div>
                    </TableCell>
                    <TableCell className="text-xs text-muted-foreground">
                      {new Date(u.created_at).toLocaleDateString("bn-BD")}
                    </TableCell>
                    <TableCell className="text-right">
                      {editingUser === u.user_id ? (
                        <div className="flex gap-1.5 justify-end">
                          <Button size="sm" variant="ghost" onClick={cancelEdit} disabled={saving}>
                            <XCircle className="h-4 w-4" />
                          </Button>
                          <Button size="sm" onClick={() => handleSave(u.user_id)} disabled={saving}>
                            {saving ? <Loader2 className="h-4 w-4 animate-spin" /> : <Save className="h-4 w-4" />}
                          </Button>
                        </div>
                      ) : (
                        <UserActions u={u} />
                      )}
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </div>

          {/* Mobile Cards */}
          <div className="md:hidden space-y-3">
            {filtered.map(u => (
              <motion.div key={u.user_id} layout
                className="bg-background rounded-xl border border-border overflow-hidden">
                <div className="p-3 flex items-center gap-3 cursor-pointer" onClick={() => toggleExpand(u.user_id)}>
                  {u.avatar_url ? (
                    <img src={u.avatar_url} alt="" className="h-10 w-10 rounded-full object-cover flex-shrink-0" />
                  ) : (
                    <div className="h-10 w-10 rounded-full bg-primary/10 flex items-center justify-center flex-shrink-0">
                      <User className="h-5 w-5 text-primary" />
                    </div>
                  )}
                  <div className="flex-1 min-w-0">
                    <div className="flex items-center gap-2">
                      <span className="font-medium text-sm truncate">{u.name || "Unknown"}</span>
                      {u.role !== "user" && (
                        <Badge className={`border-0 text-[10px] px-1.5 py-0 ${u.role === "admin" ? "bg-primary/10 text-primary" : "bg-accent text-accent-foreground"}`}>
                          {u.role === "admin" ? "Admin" : "Reseller"}
                        </Badge>
                      )}
                    </div>
                    <p className="text-xs text-muted-foreground truncate">{u.email || "—"}</p>
                  </div>
                  <div className="flex items-center gap-2">
                    <Badge variant="secondary" className="text-[10px]">{u.order_count} অর্ডার</Badge>
                    {expandedUser === u.user_id ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />}
                  </div>
                </div>

                <AnimatePresence>
                  {expandedUser === u.user_id && (
                    <motion.div
                      initial={{ height: 0, opacity: 0 }}
                      animate={{ height: "auto", opacity: 1 }}
                      exit={{ height: 0, opacity: 0 }}
                      className="border-t border-border"
                    >
                      <div className="p-3 space-y-3">
                        {editingUser === u.user_id ? (
                          <div className="space-y-2">
                            <div>
                              <Label className="text-xs">নাম</Label>
                              <Input value={editFields.name} onChange={e => setEditFields(f => ({ ...f, name: e.target.value }))}
                                className="h-8 text-sm rounded-lg" />
                            </div>
                            <div>
                              <Label className="text-xs">ইমেইল</Label>
                              <Input value={editFields.email} onChange={e => setEditFields(f => ({ ...f, email: e.target.value }))}
                                className="h-8 text-sm rounded-lg" />
                            </div>
                            <div>
                              <Label className="text-xs">ফোন</Label>
                              <Input value={editFields.phone} onChange={e => setEditFields(f => ({ ...f, phone: e.target.value }))}
                                className="h-8 text-sm rounded-lg" />
                            </div>
                            <div className="flex gap-2">
                              <Button size="sm" variant="outline" className="flex-1" onClick={cancelEdit} disabled={saving}>বাতিল</Button>
                              <Button size="sm" className="flex-1" onClick={() => handleSave(u.user_id)} disabled={saving}>
                                {saving ? <Loader2 className="h-4 w-4 animate-spin mr-1" /> : <Save className="h-4 w-4 mr-1" />}
                                সেভ
                              </Button>
                            </div>
                          </div>
                        ) : (
                          <>
                            <div className="grid grid-cols-2 gap-2 text-xs">
                              <div className="flex items-center gap-1.5 text-muted-foreground">
                                <Phone className="h-3 w-3" /> {u.phone || "—"}
                              </div>
                              <div className="flex items-center gap-1.5 text-muted-foreground">
                                <Calendar className="h-3 w-3" /> {new Date(u.created_at).toLocaleDateString("bn-BD")}
                              </div>
                            </div>
                            <div className="text-[10px] text-muted-foreground font-mono break-all">
                              ID: {u.user_id}
                            </div>
                            <div>
                              <p className="text-xs text-muted-foreground mb-1">রোল পরিবর্তন:</p>
                              <RoleSelector u={u} />
                            </div>
                            <UserActions u={u} vertical />
                          </>
                        )}
                      </div>
                    </motion.div>
                  )}
                </AnimatePresence>
              </motion.div>
            ))}
          </div>
        </SectionCard>
      )}
    </div>
  );
};

export default AdminUsersPanel;
