import { useState, useEffect } from "react";
import { supabase } from "@/integrations/supabase/client";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { useToast } from "@/hooks/use-toast";
import { SectionCard } from "./AdminShared";
import { Settings, MessageCircle, Phone, Mail, Globe, Save, Loader2, X } from "lucide-react";
import type { DBOrder } from "./adminTypes";

interface WaTemplate {
  key: string; label: string; enabled: boolean; defaultMessage: string; message: string;
}

const defaultWaTemplates: WaTemplate[] = [
  { key: "processing", label: "Processing WhatsApp", enabled: true, defaultMessage: "Dear {name} 💜\nThank you for shopping with\n{shop_name} ✨\n\nOrder: {items}\nTotal: {total}", message: "" },
  { key: "completed", label: "Completed WhatsApp", enabled: true, defaultMessage: "💜 We appreciate your trust.\n\n— {shop_name}\n\nOrder #{order_id} completed!", message: "" },
  { key: "pending", label: "Pending Payment", enabled: false, defaultMessage: "Dear {name}, Your order #{order_id} is Pending Payment.\nPlease complete your payment.", message: "" },
  { key: "cancelled", label: "Cancelled", enabled: false, defaultMessage: "Dear {name}, Your order #{order_id} is Cancelled.", message: "" },
  { key: "failed", label: "Failed", enabled: false, defaultMessage: "Dear {name}, Your order #{order_id} is Failed.", message: "" },
];

const AdminWhatsAppPanel = ({ orders }: { orders: DBOrder[] }) => {
  const [waSecret, setWaSecret] = useState("");
  const [waAccountId, setWaAccountId] = useState("");
  const [waTemplates, setWaTemplates] = useState<WaTemplate[]>(defaultWaTemplates);
  const [saving, setSaving] = useState(false);
  const [waNumber, setWaNumber] = useState("");
  const [waMessage, setWaMessage] = useState("");
  const [waSending, setWaSending] = useState(false);
  const [waMode, setWaMode] = useState<"single" | "bulk">("single");
  const [testNumber, setTestNumber] = useState("");
  const [testSending, setTestSending] = useState(false);
  const [showWaKeywords, setShowWaKeywords] = useState<string | null>(null);
  const { toast: waToast } = useToast();

  const sendWhatsApp = async (number: string, message: string) => {
    const { data, error } = await supabase.functions.invoke("send-whatsapp", { body: { number, message } });
    if (error) throw error;
    return data;
  };

  useEffect(() => {
    const load = async () => {
      const { data } = await supabase.from("site_settings").select("key, value").in("key", ["whatsapp_api_secret", "whatsapp_account_id", "whatsapp_templates"]);
      const map: Record<string, string> = {};
      data?.forEach(r => { if (r.value) map[r.key] = r.value; });
      if (map.whatsapp_api_secret) setWaSecret(map.whatsapp_api_secret);
      if (map.whatsapp_account_id) setWaAccountId(map.whatsapp_account_id);
      if (map.whatsapp_templates) {
        try {
          const saved = JSON.parse(map.whatsapp_templates) as Partial<WaTemplate>[];
          setWaTemplates(defaultWaTemplates.map(dt => {
            const found = saved.find(s => s.key === dt.key);
            return found ? { ...dt, ...found, defaultMessage: dt.defaultMessage } : dt;
          }));
        } catch {}
      }
    };
    load();
  }, []);

  const handleSave = async () => {
    setSaving(true);
    try {
      const upsert = async (key: string, value: string) => {
        const { data: existing } = await supabase.from("site_settings").select("id").eq("key", key).maybeSingle();
        if (existing) await supabase.from("site_settings").update({ value }).eq("id", existing.id);
        else await supabase.from("site_settings").insert({ key, value });
      };
      if (waSecret) await upsert("whatsapp_api_secret", waSecret);
      if (waAccountId) await upsert("whatsapp_account_id", waAccountId);
      const templateData = waTemplates.map(t => ({ key: t.key, enabled: t.enabled, message: t.message }));
      await upsert("whatsapp_templates", JSON.stringify(templateData));
      waToast({ title: "✅ WhatsApp সেটিংস সেভ হয়েছে!" });
    } catch { waToast({ title: "সেভ করতে সমস্যা হয়েছে", variant: "destructive" }); }
    finally { setSaving(false); }
  };

  const handleSendWa = async () => {
    if (!waNumber || !waMessage) { waToast({ title: "নম্বর ও মেসেজ দিন", variant: "destructive" }); return; }
    setWaSending(true);
    try { await sendWhatsApp(waNumber, waMessage); waToast({ title: "✅ WhatsApp মেসেজ পাঠানো হয়েছে!" }); setWaNumber(""); setWaMessage(""); }
    catch { waToast({ title: "WhatsApp মেসেজ পাঠাতে সমস্যা হয়েছে", variant: "destructive" }); }
    finally { setWaSending(false); }
  };

  const handleBulkWa = async () => {
    if (!waMessage) { waToast({ title: "মেসেজ দিন", variant: "destructive" }); return; }
    const uniquePhones = [...new Set(orders.filter(o => o.customer_phone).map(o => o.customer_phone!))];
    if (uniquePhones.length === 0) { waToast({ title: "কোনো কাস্টমারের ফোন নম্বর নেই", variant: "destructive" }); return; }
    setWaSending(true);
    try { for (const phone of uniquePhones) { await sendWhatsApp(phone, waMessage); } waToast({ title: `✅ ${uniquePhones.length}টি WhatsApp মেসেজ পাঠানো হয়েছে!` }); setWaMessage(""); }
    catch { waToast({ title: "কিছু মেসেজ পাঠাতে সমস্যা হয়েছে", variant: "destructive" }); }
    finally { setWaSending(false); }
  };

  const handleTestWa = async () => {
    if (!testNumber) { waToast({ title: "টেস্ট নম্বর দিন", variant: "destructive" }); return; }
    setTestSending(true);
    try { await sendWhatsApp(testNumber, "This is a test WhatsApp message. ✅"); waToast({ title: "✅ টেস্ট WhatsApp পাঠানো হয়েছে!" }); }
    catch { waToast({ title: "টেস্ট পাঠাতে সমস্যা হয়েছে", variant: "destructive" }); }
    finally { setTestSending(false); }
  };

  const updateTemplate = (key: string, field: Partial<WaTemplate>) => {
    setWaTemplates(prev => prev.map(t => t.key === key ? { ...t, ...field } : t));
  };

  const waKeywordGroups = [
    { label: "Shop", tags: ["{shop_name}", "{shop_email}", "{shop_url}", "{shop_phone}"] },
    { label: "Order", tags: ["{order_id}", "{order_status}", "{order_amount}", "{order_product}", "{order_product_with_qty}", "{order_currency}"] },
    { label: "Customer", tags: ["{name}", "{phone}", "{email}", "{billing_email}", "{billing_phone}"] },
  ];

  const insertWaTag = (templateKey: string, tag: string) => {
    const t = waTemplates.find(t => t.key === templateKey);
    if (!t) return;
    updateTemplate(templateKey, { message: (t.message || t.defaultMessage) + tag });
  };

  return (
    <div className="space-y-6 max-w-3xl">
      <SectionCard title="WhatsApp API Settings" icon={Settings}>
        <div className="space-y-3">
          <div className="space-y-1.5">
            <Label className="text-xs">API Secret</Label>
            <Input value={waSecret} onChange={e => setWaSecret(e.target.value)} placeholder="আপনার SMSQuicker API Secret" className="rounded-xl font-mono text-sm" type="password" />
          </div>
          <div className="space-y-1.5">
            <Label className="text-xs">WhatsApp Account ID</Label>
            <Input value={waAccountId} onChange={e => setWaAccountId(e.target.value)} placeholder="WhatsApp Account Unique ID" className="rounded-xl font-mono text-sm" />
          </div>
        </div>
      </SectionCard>

      <SectionCard title="WhatsApp Templates" icon={MessageCircle}>
        <div className="space-y-5">
          {waTemplates.map((template) => (
            <div key={template.key} className="border border-border rounded-xl p-3 sm:p-4 space-y-3">
              <div className="flex items-center justify-between">
                <h4 className="font-semibold text-sm">{template.label}</h4>
                <Switch checked={template.enabled} onCheckedChange={(checked) => updateTemplate(template.key, { enabled: checked })} />
              </div>
              <textarea value={template.message || template.defaultMessage} onChange={e => updateTemplate(template.key, { message: e.target.value })}
                className="flex w-full rounded-xl border border-input bg-background px-3 py-2 text-sm min-h-[100px] resize-y ring-offset-background focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring" />
              <button type="button" onClick={() => setShowWaKeywords(showWaKeywords === template.key ? null : template.key)}
                className="text-xs font-medium text-primary border border-primary/30 bg-primary/5 px-2.5 py-1 rounded-lg hover:bg-primary/10 transition-colors">
                Keywords
              </button>
              {showWaKeywords === template.key && (
                <div className="border border-border rounded-xl p-3 space-y-3 bg-muted/30 relative">
                  <button type="button" onClick={() => setShowWaKeywords(null)} className="absolute top-2 right-2 h-6 w-6 rounded-full bg-muted flex items-center justify-center">
                    <X className="h-3.5 w-3.5" />
                  </button>
                  {waKeywordGroups.map(group => (
                    <div key={group.label}>
                      <p className="text-xs font-semibold mb-1.5">{group.label}</p>
                      <div className="flex flex-wrap gap-1.5">
                        {group.tags.map(tag => (
                          <button key={tag} type="button" onClick={() => insertWaTag(template.key, tag)}
                            className="text-[11px] font-mono text-primary bg-primary/10 border border-primary/20 px-2 py-1 rounded-md hover:bg-primary/20 transition-colors">
                            {tag}
                          </button>
                        ))}
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </div>
          ))}
          <Button onClick={handleSave} disabled={saving} className="gap-1.5 rounded-xl w-full sm:w-auto">
            {saving ? <Loader2 className="h-4 w-4 animate-spin" /> : <Save className="h-4 w-4" />}
            {saving ? "সেভ হচ্ছে..." : "Save Changes"}
          </Button>
        </div>
      </SectionCard>

      <SectionCard title="Send Test WhatsApp" icon={Phone}>
        <div className="flex flex-col sm:flex-row gap-2 sm:items-end">
          <div className="flex-1 space-y-1.5">
            <Input value={testNumber} onChange={e => setTestNumber(e.target.value)} placeholder="017XXXXXXXX" className="rounded-xl" />
          </div>
          <Button onClick={handleTestWa} disabled={testSending} className="gap-1.5 rounded-xl w-full sm:w-auto">
            {testSending ? <Loader2 className="h-4 w-4 animate-spin" /> : <Phone className="h-4 w-4" />}
            Send Test
          </Button>
        </div>
      </SectionCard>

      <SectionCard title="WhatsApp পাঠান" icon={Phone}>
        <div className="flex gap-2 mb-6">
          <button onClick={() => setWaMode("single")}
            className={`px-3 sm:px-4 py-2 rounded-xl text-sm font-medium transition-all ${waMode === "single" ? "bg-primary text-primary-foreground" : "bg-muted text-muted-foreground"}`}>
            একজনকে পাঠান
          </button>
          <button onClick={() => setWaMode("bulk")}
            className={`px-3 sm:px-4 py-2 rounded-xl text-sm font-medium transition-all ${waMode === "bulk" ? "bg-primary text-primary-foreground" : "bg-muted text-muted-foreground"}`}>
            সবাইকে পাঠান
          </button>
        </div>
        <div className="space-y-4">
          {waMode === "single" && (
            <div className="space-y-1.5">
              <Label className="text-xs"><Phone className="h-3 w-3 inline mr-1" />ফোন নম্বর</Label>
              <Input value={waNumber} onChange={e => setWaNumber(e.target.value)} placeholder="01XXXXXXXXX" className="rounded-xl" />
            </div>
          )}
          {waMode === "bulk" && (
            <div className="rounded-xl bg-muted/50 border border-border p-4">
              <p className="text-sm text-muted-foreground">
                <span className="font-semibold text-foreground">{[...new Set(orders.filter(o => o.customer_phone).map(o => o.customer_phone!))].length}</span> জন কাস্টমারকে WhatsApp পাঠানো হবে
              </p>
            </div>
          )}
          <div className="space-y-1.5">
            <Label className="text-xs"><Mail className="h-3 w-3 inline mr-1" />মেসেজ</Label>
            <textarea value={waMessage} onChange={e => setWaMessage(e.target.value)} placeholder="আপনার মেসেজ লিখুন..."
              className="flex w-full rounded-xl border border-input bg-background px-3 py-2 text-sm min-h-[120px] resize-y ring-offset-background focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring" />
          </div>
          <Button onClick={waMode === "single" ? handleSendWa : handleBulkWa} disabled={waSending} className="gap-1.5 rounded-xl w-full sm:w-auto">
            {waSending ? <Loader2 className="h-4 w-4 animate-spin" /> : <Phone className="h-4 w-4" />}
            {waSending ? "পাঠানো হচ্ছে..." : waMode === "single" ? "WhatsApp পাঠান" : "সবাইকে WhatsApp পাঠান"}
          </Button>
        </div>
      </SectionCard>

      <SectionCard title="WhatsApp টেমপ্লেট" icon={Globe}>
        <div className="space-y-3">
          {[
            { label: "অর্ডার কনফার্মেশন", msg: "প্রিয় গ্রাহক, আপনার অর্ডার সফলভাবে গ্রহণ করা হয়েছে। ধন্যবাদ! 🎉" },
            { label: "অর্ডার সম্পন্ন", msg: "প্রিয় গ্রাহক, আপনার অর্ডার সফলভাবে সম্পন্ন হয়েছে। ধন্যবাদ! ✅" },
            { label: "নতুন অফার", msg: "🎉 বিশেষ অফার! সীমিত সময়ের জন্য সকল প্রোডাক্টে বিশেষ ছাড়। এখনই অর্ডার করুন! 🛒" },
            { label: "পেমেন্ট রিমাইন্ডার", msg: "প্রিয় গ্রাহক, আপনার অর্ডারের পেমেন্ট এখনো সম্পন্ন হয়নি। দ্রুত পেমেন্ট সম্পন্ন করুন। ⏰" },
          ].map((t, i) => (
            <button key={i} onClick={() => setWaMessage(t.msg)}
              className="w-full text-left p-3 rounded-xl border border-border hover:bg-muted/50 transition-colors">
              <p className="text-sm font-medium text-foreground">{t.label}</p>
              <p className="text-xs text-muted-foreground mt-0.5 line-clamp-1">{t.msg}</p>
            </button>
          ))}
        </div>
      </SectionCard>
    </div>
  );
};

export default AdminWhatsAppPanel;
