import { createContext, useContext, useEffect, useState, ReactNode } from "react";
import { supabase } from "@/integrations/supabase/client";
import type { User, Session } from "@supabase/supabase-js";

type UserRole = "admin" | "reseller" | "user";

interface AuthContextType {
  user: User | null;
  session: Session | null;
  loading: boolean;
  role: UserRole;
  isReseller: boolean;
  isAdmin: boolean;
  signOut: () => Promise<void>;
}

const AuthContext = createContext<AuthContextType>({
  user: null,
  session: null,
  loading: true,
  role: "user",
  isReseller: false,
  isAdmin: false,
  signOut: async () => {},
});

export const useAuth = () => useContext(AuthContext);

export const AuthProvider = ({ children }: { children: ReactNode }) => {
  const [user, setUser] = useState<User | null>(null);
  const [session, setSession] = useState<Session | null>(null);
  const [loading, setLoading] = useState(true);
  const [role, setRole] = useState<UserRole>("user");

  const fetchRole = async (userId: string) => {
    const { data } = await supabase
      .from("user_roles")
      .select("role")
      .eq("user_id", userId);
    
    if (data && data.length > 0) {
      // Priority: admin > reseller > user
      if (data.some(r => r.role === "admin")) setRole("admin");
      else if (data.some(r => r.role === "reseller")) setRole("reseller");
      else setRole("user");
    } else {
      setRole("user");
    }
  };

  useEffect(() => {
    const syncGoogleProfile = async (u: User) => {
      const provider = u.app_metadata?.provider;
      if (provider === "google") {
        const meta = u.user_metadata;
        const name = meta?.full_name || meta?.name || null;
        const avatar = meta?.avatar_url || meta?.picture || null;
        if (name || avatar) {
          await supabase.from("profiles").update({
            ...(name ? { name } : {}),
            ...(avatar ? { avatar_url: avatar } : {}),
            email: u.email,
          }).eq("user_id", u.id);
        }
      }
    };

    const { data: { subscription } } = supabase.auth.onAuthStateChange((_event, session) => {
      setSession(session);
      setUser(session?.user ?? null);
      setLoading(false);
      if (session?.user) {
        setTimeout(() => {
          fetchRole(session.user.id);
          syncGoogleProfile(session.user);
        }, 0);
      } else {
        setRole("user");
      }
    });

    supabase.auth.getSession().then(({ data: { session } }) => {
      setSession(session);
      setUser(session?.user ?? null);
      setLoading(false);
      if (session?.user) {
        fetchRole(session.user.id);
      }
    });

    return () => subscription.unsubscribe();
  }, []);

  const signOut = async () => {
    await supabase.auth.signOut();
    setRole("user");
  };

  return (
    <AuthContext.Provider value={{ 
      user, session, loading, role,
      isReseller: role === "reseller" || role === "admin",
      isAdmin: role === "admin",
      signOut 
    }}>
      {children}
    </AuthContext.Provider>
  );
};
