import { createContext, useContext, useState, type ReactNode } from "react";
import type { Product } from "@/components/productsData";

export interface CartItem {
  product: Product;
  selectedOption: string | null;
  quantity: number;
  customFieldValues?: Record<string, string>;
}

interface CartContextType {
  items: CartItem[];
  addToCart: (product: Product, selectedOption: string | null, quantity: number, customFieldValues?: Record<string, string>) => void;
  removeFromCart: (productId: number | string, selectedOption: string | null) => void;
  updateQuantity: (productId: number | string, selectedOption: string | null, quantity: number) => void;
  clearCart: () => void;
  totalItems: number;
  isCartOpen: boolean;
  setIsCartOpen: (open: boolean) => void;
}

const CartContext = createContext<CartContextType | undefined>(undefined);

export const CartProvider = ({ children }: { children: ReactNode }) => {
  const [items, setItems] = useState<CartItem[]>([]);
  const [isCartOpen, setIsCartOpen] = useState(false);

  const addToCart = (product: Product, selectedOption: string | null, quantity: number, customFieldValues?: Record<string, string>) => {
    setItems((prev) => {
      const existing = prev.find(
        (item) => item.product.id === product.id && item.selectedOption === selectedOption
      );
      if (existing) {
        return prev.map((item) =>
          item.product.id === product.id && item.selectedOption === selectedOption
            ? { ...item, quantity: item.quantity + quantity, customFieldValues: customFieldValues || item.customFieldValues }
            : item
        );
      }
      return [...prev, { product, selectedOption, quantity, customFieldValues }];
    });
    setIsCartOpen(true);
  };

  const removeFromCart = (productId: number | string, selectedOption: string | null) => {
    setItems((prev) =>
      prev.filter((item) => !(item.product.id === productId && item.selectedOption === selectedOption))
    );
  };

  const updateQuantity = (productId: number | string, selectedOption: string | null, quantity: number) => {
    if (quantity <= 0) {
      removeFromCart(productId, selectedOption);
      return;
    }
    setItems((prev) =>
      prev.map((item) =>
        item.product.id === productId && item.selectedOption === selectedOption
          ? { ...item, quantity }
          : item
      )
    );
  };

  const clearCart = () => setItems([]);

  const totalItems = items.reduce((sum, item) => sum + item.quantity, 0);

  return (
    <CartContext.Provider
      value={{ items, addToCart, removeFromCart, updateQuantity, clearCart, totalItems, isCartOpen, setIsCartOpen }}
    >
      {children}
    </CartContext.Provider>
  );
};

export const useCart = () => {
  const context = useContext(CartContext);
  if (!context) throw new Error("useCart must be used within CartProvider");
  return context;
};
