import { useState, useEffect } from "react";
import { supabase } from "@/integrations/supabase/client";

export interface SiteSettings {
  logo_url: string | null;
  site_title: string;
  site_description: string;
  favicon_url: string | null;
  phone: string;
  email: string;
  facebook_url: string;
  whatsapp_number: string;
  address: string;
  developer_name: string;
  developer_url: string;
  payment_banner_url: string | null;
  trust_badge_url: string | null;
}

const defaults: SiteSettings = {
  logo_url: null,
  site_title: "TZSS",
  site_description: "Unlock unlimited Digital Entertainment",
  favicon_url: null,
  phone: "+880 1851-182700",
  email: "info@tzssbd.com",
  facebook_url: "",
  whatsapp_number: "",
  address: "Dhaka, Bangladesh",
  developer_name: "TEAM RONY",
  developer_url: "https://hanifahmedrony.site",
  payment_banner_url: null,
  trust_badge_url: null,
};

export const useSiteSettings = () => {
  const [settings, setSettings] = useState<SiteSettings>(defaults);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchSettings = async () => {
      const { data } = await supabase.from("site_settings").select("key, value");
      if (data && data.length > 0) {
        const merged = { ...defaults };
        data.forEach((row) => {
          if (row.key in merged) {
            (merged as any)[row.key] = row.value || (defaults as any)[row.key];
          }
        });
        setSettings(merged);
      }
      setLoading(false);
    };
    fetchSettings();
  }, []);

  // Apply dynamic title & favicon
  useEffect(() => {
    if (loading) return;
    document.title = settings.site_title || defaults.site_title;

    // Update meta description
    const metaDesc = document.querySelector('meta[name="description"]');
    if (metaDesc) metaDesc.setAttribute("content", settings.site_description || defaults.site_description);

    // Update favicon
    if (settings.favicon_url) {
      let link = document.querySelector("link[rel='icon']") as HTMLLinkElement;
      if (!link) {
        link = document.createElement("link");
        link.rel = "icon";
        document.head.appendChild(link);
      }
      link.href = settings.favicon_url;
      link.type = "image/png";
    }
  }, [settings, loading]);

  return { settings, loading };
};
