import { useState, useEffect } from "react";
import { useParams, Link } from "react-router-dom";
import TopBar from "@/components/TopBar";
import Header from "@/components/Header";
import ShopFooter from "@/components/ShopFooter";
import BottomBar from "@/components/BottomBar";
import ShopProductCard from "@/components/ShopProductCard";
import { useSiteSettings } from "@/hooks/useSiteSettings";
import { supabase } from "@/integrations/supabase/client";
import type { Product } from "@/components/productsData";
import { ChevronRight, Home, Search } from "lucide-react";
import { motion, AnimatePresence } from "framer-motion";
import { Input } from "@/components/ui/input";
import { X } from "lucide-react";

const Category = () => {
  useSiteSettings();
  const { slug } = useParams<{ slug: string }>();
  const [products, setProducts] = useState<Product[]>([]);
  const [categoryName, setCategoryName] = useState("");
  const [loading, setLoading] = useState(true);
  const [searchQuery, setSearchQuery] = useState("");

  useEffect(() => {
    const fetchProducts = async () => {
      setLoading(true);
      // Fetch all categories to find matching one
      const { data: cats } = await supabase
        .from("categories")
        .select("*")
        .eq("is_active", true)
        .order("sort_order");

      const decodedSlug = decodeURIComponent(slug || "");
      const matchedCat = cats?.find(
        (c) => c.name.toLowerCase().replace(/\s+/g, "-") === decodedSlug.toLowerCase() ||
               c.name === decodedSlug
      );

      if (matchedCat) {
        setCategoryName(matchedCat.name);
        const { data: prods } = await supabase
          .from("products")
          .select("*")
          .ilike("category", `%${matchedCat.name}%`);

        if (prods) {
          setProducts(
            prods.map((p: any) => ({
              id: p.id,
              name: p.name,
              slug: p.slug || p.id,
              image: p.image_url || "/placeholder.svg",
              price: p.price,
              originalPrice: p.original_price || undefined,
              resellerPrice: p.reseller_price || undefined,
              resellerPrices: p.reseller_prices || undefined,
              category: p.category,
              badges: (p.badges || []) as ("sale" | "limited")[],
              description: p.description || undefined,
              features: p.features || undefined,
              duration: p.duration_options || undefined,
            }))
          );
        }
      }
      setLoading(false);
    };
    fetchProducts();
  }, [slug]);

  const filteredProducts = searchQuery.trim()
    ? products.filter(
        (p) =>
          p.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
          p.description?.toLowerCase().includes(searchQuery.toLowerCase())
      )
    : products;

  return (
    <div className="min-h-screen bg-background">
      <TopBar />
      <Header />

      {/* Breadcrumb + Hero */}
      <div className="bg-gradient-to-br from-primary/10 via-background to-accent/5 border-b border-border">
        <div className="container mx-auto px-4 py-6 md:py-10">
          {/* Breadcrumb */}
          <div className="flex items-center gap-1.5 text-sm text-muted-foreground mb-4">
            <Link to="/" className="hover:text-primary transition-colors flex items-center gap-1">
              <Home className="h-3.5 w-3.5" />
              হোম
            </Link>
            <ChevronRight className="h-3.5 w-3.5" />
            <Link to="/shop" className="hover:text-primary transition-colors">শপ</Link>
            <ChevronRight className="h-3.5 w-3.5" />
            <span className="text-foreground font-medium">{categoryName || "ক্যাটাগরি"}</span>
          </div>

          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.5 }}
            className="text-center max-w-2xl mx-auto"
          >
            <h1 className="font-heading text-2xl md:text-4xl font-bold text-foreground mb-2">
              {categoryName || "ক্যাটাগরি"}
            </h1>
            <p className="text-muted-foreground text-sm md:text-base mb-5">
              {categoryName} ক্যাটাগরির সকল প্রোডাক্ট দেখুন
            </p>

            {/* Search */}
            <div className="relative max-w-md mx-auto">
              <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
              <Input
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                placeholder="এই ক্যাটাগরিতে খুঁজুন..."
                className="pl-10 pr-10 h-11 rounded-full bg-card border-border"
              />
              {searchQuery && (
                <button
                  onClick={() => setSearchQuery("")}
                  className="absolute right-3 top-1/2 -translate-y-1/2 text-muted-foreground hover:text-foreground"
                >
                  <X className="h-4 w-4" />
                </button>
              )}
            </div>
          </motion.div>
        </div>
      </div>

      {/* Products */}
      <div className="container mx-auto px-4 py-6 md:py-8">
        <p className="text-sm text-muted-foreground mb-4">
          {filteredProducts.length}টি প্রোডাক্ট পাওয়া গেছে
          {searchQuery && ` — "${searchQuery}"`}
        </p>

        {loading ? (
          <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 gap-3 md:gap-4">
            {[...Array(10)].map((_, i) => (
              <div key={i} className="bg-card rounded-xl border border-border overflow-hidden animate-pulse">
                <div className="aspect-[4/3] bg-muted" />
                <div className="p-3 space-y-2">
                  <div className="h-3 bg-muted rounded w-3/4" />
                  <div className="h-4 bg-muted rounded w-1/2" />
                </div>
              </div>
            ))}
          </div>
        ) : (
          <AnimatePresence mode="wait">
            {filteredProducts.length > 0 ? (
              <motion.div
                key={searchQuery}
                initial={{ opacity: 0 }}
                animate={{ opacity: 1 }}
                exit={{ opacity: 0 }}
                transition={{ duration: 0.2 }}
                className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 gap-3 md:gap-4"
              >
                {filteredProducts.map((product, idx) => (
                  <ShopProductCard key={product.id} product={product} index={idx} />
                ))}
              </motion.div>
            ) : (
              <motion.div
                initial={{ opacity: 0 }}
                animate={{ opacity: 1 }}
                className="text-center py-20"
              >
                <Search className="h-12 w-12 text-muted-foreground/40 mx-auto mb-4" />
                <h3 className="font-heading font-semibold text-lg mb-1">কোনো প্রোডাক্ট পাওয়া যায়নি</h3>
                <p className="text-sm text-muted-foreground">এই ক্যাটাগরিতে কোনো প্রোডাক্ট নেই</p>
              </motion.div>
            )}
          </AnimatePresence>
        )}
      </div>

      <ShopFooter />
      <BottomBar />
    </div>
  );
};

export default Category;
