import { useState, useEffect, useRef, useCallback } from "react";
import { useNavigate } from "react-router-dom";
import { trackPixelEvent } from "@/hooks/usePixel";
import { useCart } from "@/contexts/CartContext";
import { supabase } from "@/integrations/supabase/client";
import { useAuth } from "@/contexts/AuthContext";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { useToast } from "@/hooks/use-toast";
import TopBar from "@/components/TopBar";
import Header from "@/components/Header";
import ShopFooter from "@/components/ShopFooter";
import BottomBar from "@/components/BottomBar";
import { ShoppingBag, ArrowLeft, CheckCircle2, Loader2, Tag, X } from "lucide-react";
import { z } from "zod";
import { parseDurationOption } from "@/lib/priceUtils";

const checkoutSchema = z.object({
  name: z.string().trim().min(2, "নাম কমপক্ষে ২ অক্ষর হতে হবে").max(100),
  email: z.string().trim().email("সঠিক ইমেইল দিন"),
  phone: z.string().trim().min(11, "সঠিক ফোন নম্বর দিন").max(15),
});

interface AppliedPromo {
  id: string;
  code: string;
  discount_type: string;
  discount_value: number;
}

const Checkout = () => {
  const { user, isReseller } = useAuth();
  const { items, clearCart, setIsCartOpen } = useCart();
  const [name, setName] = useState("");
  const [email, setEmail] = useState("");
  const [phone, setPhone] = useState("");
  const [loading, setLoading] = useState(false);
  const [orderPlaced, setOrderPlaced] = useState(false);
  const [errors, setErrors] = useState<Record<string, string>>({});
  const [promoCode, setPromoCode] = useState("");
  const [promoLoading, setPromoLoading] = useState(false);
  const [appliedPromo, setAppliedPromo] = useState<AppliedPromo | null>(null);
  const [promoError, setPromoError] = useState("");
  const navigate = useNavigate();
  const { toast } = useToast();
  const sessionIdRef = useRef<string>("");
  const debounceRef = useRef<ReturnType<typeof setTimeout>>();
  const incompleteIdRef = useRef<string | null>(null);

  // Redirect to login if not authenticated (guest checkout disabled)
  useEffect(() => {
    if (!user) {
      toast({ title: "লগইন করুন", description: "অর্ডার করতে আগে লগইন করতে হবে।", variant: "destructive" });
      navigate("/auth", { state: { from: "/checkout" } });
    }
  }, [user, navigate]);

  // Generate a unique session ID for this checkout visit
  useEffect(() => {
    sessionIdRef.current = `${Date.now()}-${Math.random().toString(36).slice(2, 9)}`;
  }, []);

  // Parse price from string like "৳299" or "299"
  const parsePrice = (price: string): number => {
    const num = price.replace(/[^\d.]/g, "");
    return parseFloat(num) || 0;
  };

  const getItemPrice = (item: typeof items[0]) => {
    if (isReseller && item.selectedOption && item.product.resellerPrices?.[item.selectedOption]) {
      return item.product.resellerPrices[item.selectedOption];
    }
    if (isReseller && item.product.resellerPrice) {
      return item.product.resellerPrice;
    }
    if (item.selectedOption) {
      const parsed = parseDurationOption(item.selectedOption);
      if (parsed.price) return parsed.price;
    }
    return item.product.price;
  };

  const subtotal = items.reduce((sum, item) => sum + parsePrice(getItemPrice(item)) * item.quantity, 0);

  // Debounced save of incomplete order data
  const saveIncompleteOrder = useCallback(() => {
    if (debounceRef.current) clearTimeout(debounceRef.current);
    debounceRef.current = setTimeout(async () => {
      const hasAnyInput = name.trim() || email.trim() || phone.trim();
      if (!hasAnyInput && items.length === 0) return;

      const cartData = items.map((item) => ({
        name: item.product.name,
        option: item.selectedOption,
        quantity: item.quantity,
        price: getItemPrice(item),
      }));

      const payload = {
        session_id: sessionIdRef.current,
        name: name.trim(),
        email: email.trim(),
        phone: phone.trim(),
        cart_items: cartData,
        cart_total: subtotal,
        updated_at: new Date().toISOString(),
      };

      if (incompleteIdRef.current) {
        await supabase
          .from("incomplete_orders")
          .update(payload as any)
          .eq("id", incompleteIdRef.current);
      } else {
        const { data } = await supabase
          .from("incomplete_orders")
          .insert(payload as any)
          .select("id")
          .single();
        if (data) incompleteIdRef.current = (data as any).id;
      }
    }, 1500);
  }, [name, email, phone, items, subtotal]);

  // Trigger save when form fields change
  useEffect(() => {
    saveIncompleteOrder();
    return () => {
      if (debounceRef.current) clearTimeout(debounceRef.current);
    };
  }, [name, email, phone]);

  // Track InitiateCheckout on mount
  useEffect(() => {
    if (items.length > 0) {
      trackPixelEvent("InitiateCheckout", { value: items.reduce((sum, item) => sum + (parseFloat(item.product.price.replace(/[^\d.]/g, "")) || 0) * item.quantity, 0), currency: "BDT", num_items: items.length });
    }
  }, []);

  const discountAmount = appliedPromo
    ? appliedPromo.discount_type === "percentage"
      ? Math.round(subtotal * appliedPromo.discount_value / 100)
      : Math.min(appliedPromo.discount_value, subtotal)
    : 0;

  const total = subtotal - discountAmount;

  const handleApplyPromo = async () => {
    const code = promoCode.trim().toUpperCase();
    if (!code) return;
    setPromoError("");
    setPromoLoading(true);

    const { data, error } = await supabase
      .from("promo_codes" as any)
      .select("*")
      .eq("code", code)
      .eq("is_active", true)
      .maybeSingle();

    if (error || !data) {
      setPromoError("প্রোমো কোড পাওয়া যায়নি");
      setPromoLoading(false);
      return;
    }

    const promo = data as any;

    // Check expiry
    if (promo.expires_at && new Date(promo.expires_at) < new Date()) {
      setPromoError("প্রোমো কোডের মেয়াদ শেষ");
      setPromoLoading(false);
      return;
    }

    // Check max uses
    if (promo.max_uses && promo.used_count >= promo.max_uses) {
      setPromoError("প্রোমো কোডের ব্যবহার সীমা শেষ");
      setPromoLoading(false);
      return;
    }

    // Check min order amount
    if (promo.min_order_amount && subtotal < promo.min_order_amount) {
      setPromoError(`মিনিমাম ৳${promo.min_order_amount} অর্ডার প্রয়োজন`);
      setPromoLoading(false);
      return;
    }

    // Check per-user usage limit
    if (user?.id) {
      const { data: usageData } = await supabase
        .from("promo_usage" as any)
        .select("id")
        .eq("promo_code_id", promo.id)
        .eq("user_id", user.id);
      if (usageData && (usageData as any[]).length > 0) {
        setPromoError("আপনি ইতিমধ্যে এই প্রোমো কোড ব্যবহার করেছেন");
        setPromoLoading(false);
        return;
      }
    }

    setAppliedPromo({
      id: promo.id,
      code: promo.code,
      discount_type: promo.discount_type,
      discount_value: promo.discount_value,
    });
    setPromoLoading(false);
    toast({ title: "✅ প্রোমো কোড প্রয়োগ হয়েছে!" });
  };

  const handleRemovePromo = () => {
    setAppliedPromo(null);
    setPromoCode("");
    setPromoError("");
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setErrors({});

    const result = checkoutSchema.safeParse({ name, email, phone });
    if (!result.success) {
      const fieldErrors: Record<string, string> = {};
      result.error.errors.forEach((err) => {
        if (err.path[0]) fieldErrors[err.path[0] as string] = err.message;
      });
      setErrors(fieldErrors);
      return;
    }

    if (items.length === 0) {
      toast({ title: "কার্ট খালি!", variant: "destructive" });
      return;
    }

    setLoading(true);

    try {
      const callbackUrl = `${window.location.origin}/payment-callback`;

      const paymentItems = items.map((item) => {
        let productName = `${item.product.name}${item.selectedOption ? ` (${item.selectedOption})` : ""} x${item.quantity}`;
        // Append custom field values
        if (item.customFieldValues && Object.keys(item.customFieldValues).length > 0) {
          const cfInfo = Object.entries(item.customFieldValues).map(([k, v]) => `${k}: ${v}`).join(", ");
          productName += ` [${cfInfo}]`;
        }
        return {
          product_name: productName,
          amount: getItemPrice(item),
          quantity: item.quantity,
        };
      });

      const { data, error } = await supabase.functions.invoke("tzsmmpay-create", {
        body: {
          customer_name: result.data.name,
          customer_phone: result.data.phone,
          customer_email: result.data.email,
          items: paymentItems,
          callback_url: callbackUrl,
          user_id: user?.id || null,
          promo_code_id: appliedPromo?.id || null,
          discount_amount: discountAmount,
        },
      });

      if (error || !data?.payment_url) {
        toast({
          title: "পেমেন্ট শুরু করতে সমস্যা হয়েছে",
          description: data?.error || error?.message || "আবার চেষ্টা করুন",
          variant: "destructive",
        });
        setLoading(false);
        return;
      }

      // Increment promo usage & record in promo_usage
      if (appliedPromo) {
        const { data: promoData } = await supabase
          .from("promo_codes" as any)
          .select("used_count")
          .eq("id", appliedPromo.id)
          .single();
        if (promoData) {
          await supabase
            .from("promo_codes" as any)
            .update({ used_count: ((promoData as any).used_count || 0) + 1 } as any)
            .eq("id", appliedPromo.id);
        }
        // Record per-user usage
        await supabase.from("promo_usage" as any).insert({
          promo_code_id: appliedPromo.id,
          user_id: user?.id || null,
          discount_amount: discountAmount,
        } as any);
      }

      // Mark incomplete order as converted
      if (incompleteIdRef.current) {
        await supabase
          .from("incomplete_orders")
          .update({ is_converted: true } as any)
          .eq("id", incompleteIdRef.current);
      }

      clearCart();
      setIsCartOpen(false);
      window.location.href = data.payment_url;
    } catch (err) {
      toast({ title: "কিছু ভুল হয়েছে", variant: "destructive" });
      setLoading(false);
    }
  };

  if (orderPlaced) {
    return (
      <div className="min-h-screen bg-background">
        <TopBar />
        <Header />
        <div className="container mx-auto px-4 py-20 text-center">
          <div className="max-w-md mx-auto">
            <CheckCircle2 className="h-20 w-20 text-primary mx-auto mb-4" />
            <h1 className="font-heading text-2xl font-bold mb-2">অর্ডার সফল হয়েছে! 🎉</h1>
            <p className="text-muted-foreground mb-6">
              আপনার অর্ডার গ্রহণ করা হয়েছে। আমরা শীঘ্রই আপনার সাথে যোগাযোগ করবো।
            </p>
            <Button onClick={() => navigate("/")} className="gap-2">
              <ArrowLeft className="h-4 w-4" /> হোম পেজে যান
            </Button>
          </div>
        </div>
        <ShopFooter />
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-background">
      <TopBar />
      <Header />
      <div className="container mx-auto px-3 sm:px-4 py-4 sm:py-8">
        <div className="max-w-2xl mx-auto">
          <button onClick={() => navigate(-1)} className="hidden sm:flex items-center gap-1 text-sm text-muted-foreground hover:text-foreground mb-6 transition-colors">
            <ArrowLeft className="h-4 w-4" /> ফিরে যান
          </button>

          <h1 className="font-heading text-xl sm:text-2xl font-bold mb-4 sm:mb-6 flex items-center gap-2">
            <ShoppingBag className="h-5 w-5 sm:h-6 sm:w-6 text-primary" /> Checkout
            {isReseller && (
              <span className="text-xs font-semibold bg-primary/10 text-primary px-2.5 py-1 rounded-full">🏪 রিসেলার প্রাইস</span>
            )}
          </h1>

          <div className="grid gap-4 sm:gap-6 md:grid-cols-5">
            {/* Form */}
            <div className="md:col-span-3">
              <form onSubmit={handleSubmit} className="bg-card rounded-xl border border-border p-4 sm:p-5 space-y-4">
                <h2 className="font-heading font-semibold text-sm border-b border-border pb-3">আপনার তথ্য</h2>
                <div className="space-y-2">
                  <Label htmlFor="name">নাম *</Label>
                  <Input id="name" value={name} onChange={(e) => setName(e.target.value)} placeholder="আপনার নাম" maxLength={100} />
                  {errors.name && <p className="text-xs text-destructive">{errors.name}</p>}
                </div>
                <div className="space-y-2">
                  <Label htmlFor="email">ইমেইল *</Label>
                  <Input id="email" type="email" value={email} onChange={(e) => setEmail(e.target.value)} placeholder="example@email.com" maxLength={255} />
                  {errors.email && <p className="text-xs text-destructive">{errors.email}</p>}
                </div>
                <div className="space-y-2">
                  <Label htmlFor="phone">ফোন নম্বর *</Label>
                  <Input id="phone" type="tel" value={phone} onChange={(e) => setPhone(e.target.value)} placeholder="01XXXXXXXXX" maxLength={15} />
                  {errors.phone && <p className="text-xs text-destructive">{errors.phone}</p>}
                </div>

                {/* Promo Code */}
                <div className="space-y-2">
                  <Label className="flex items-center gap-1.5 text-sm">
                    <Tag className="h-3.5 w-3.5" /> প্রোমো কোড
                  </Label>
                  {appliedPromo ? (
                    <div className="flex items-center gap-2 bg-primary/10 border border-primary/20 rounded-lg px-3 py-2">
                      <Tag className="h-4 w-4 text-primary" />
                      <span className="text-sm font-semibold text-primary flex-1">{appliedPromo.code}</span>
                      <span className="text-xs text-primary">
                        {appliedPromo.discount_type === "percentage" ? `${appliedPromo.discount_value}% ছাড়` : `৳${appliedPromo.discount_value} ছাড়`}
                      </span>
                      <button type="button" onClick={handleRemovePromo} className="p-0.5 hover:bg-primary/20 rounded transition-colors">
                        <X className="h-4 w-4 text-primary" />
                      </button>
                    </div>
                  ) : (
                    <div className="flex gap-2">
                      <Input
                        value={promoCode}
                        onChange={(e) => { setPromoCode(e.target.value.toUpperCase()); setPromoError(""); }}
                        placeholder="কোড লিখুন"
                        className="uppercase"
                        maxLength={30}
                      />
                      <Button type="button" variant="outline" size="sm" className="px-4 h-10 shrink-0" onClick={handleApplyPromo} disabled={promoLoading || !promoCode.trim()}>
                        {promoLoading ? <Loader2 className="h-4 w-4 animate-spin" /> : "প্রয়োগ"}
                      </Button>
                    </div>
                  )}
                  {promoError && <p className="text-xs text-destructive">{promoError}</p>}
                </div>

                {/* Payment info */}
                <div className="bg-muted/50 rounded-lg p-3 border border-border">
                  <p className="text-xs text-muted-foreground flex items-center gap-1.5">
                    🔒 পেমেন্ট নিরাপদভাবে প্রসেস করা হবে
                  </p>
                </div>

                <Button type="submit" className="w-full h-11 font-semibold gap-2" disabled={loading || items.length === 0}>
                  {loading ? <><Loader2 className="h-4 w-4 animate-spin" /> পেমেন্ট পেজে যাচ্ছে...</> : `পেমেন্ট করুন — ৳${total}`}
                </Button>
              </form>
            </div>

            {/* Order Summary */}
            <div className="md:col-span-2">
              <div className="bg-card rounded-xl border border-border p-4 sm:p-5 space-y-4 sticky top-24">
                <h2 className="font-heading font-semibold text-sm border-b border-border pb-3">অর্ডার সামারি</h2>
                {items.length === 0 ? (
                  <p className="text-sm text-muted-foreground text-center py-4">কার্ট খালি</p>
                ) : (
                  <div className="space-y-3">
                    {items.map((item, idx) => (
                      <div key={idx} className="flex gap-3">
                        <img src={item.product.image} alt={item.product.name} className="h-12 w-12 rounded-lg object-contain bg-muted p-1" />
                        <div className="flex-1 min-w-0">
                          <p className="text-sm font-medium line-clamp-1">{item.product.name}</p>
                          {item.selectedOption && <p className="text-xs text-muted-foreground">{item.selectedOption}</p>}
                          {item.customFieldValues && Object.entries(item.customFieldValues).map(([k, v]) => (
                            <p key={k} className="text-xs text-muted-foreground">{k}: {v}</p>
                          ))}
                          <div className="flex justify-between items-center mt-0.5">
                            <span className="text-xs text-muted-foreground">x{item.quantity}</span>
                            <div className="flex items-center gap-1.5">
                              {isReseller && item.product.resellerPrice && (
                                <span className="text-xs line-through text-muted-foreground">{item.product.price}</span>
                              )}
                              <span className="text-sm font-semibold text-primary">{getItemPrice(item)}</span>
                            </div>
                          </div>
                        </div>
                      </div>
                    ))}
                    <div className="border-t border-border pt-3 space-y-1.5">
                      <div className="flex justify-between text-sm">
                        <span className="text-muted-foreground">সাবটোটাল</span>
                        <span>৳{subtotal}</span>
                      </div>
                      {discountAmount > 0 && (
                        <div className="flex justify-between text-sm text-primary">
                          <span>ডিসকাউন্ট ({appliedPromo?.code})</span>
                          <span>-৳{discountAmount}</span>
                        </div>
                      )}
                      <div className="flex justify-between font-heading font-semibold text-base border-t border-border pt-2">
                        <span>মোট</span>
                        <span className="text-primary">৳{total}</span>
                      </div>
                    </div>
                  </div>
                )}
              </div>
            </div>
          </div>
        </div>
      </div>
      <ShopFooter />
      <BottomBar />
    </div>
  );
};

export default Checkout;
