import { useEffect, useState } from "react";
import TopBar from "@/components/TopBar";
import { usePixelPageTracking, usePixelInternalClickTracking } from "@/hooks/usePixel";
import Header from "@/components/Header";
import HeroBanner from "@/components/HeroBanner";
import ProductSection from "@/components/ProductSection";
import ShopFooter from "@/components/ShopFooter";
import PartnersSection from "@/components/PartnersSection";
import AppInstallBanner from "@/components/AppInstallBanner";
import PopupModal from "@/components/PopupModal";
import BottomBar from "@/components/BottomBar";
import { useSiteSettings } from "@/hooks/useSiteSettings";
import { supabase } from "@/integrations/supabase/client";
import type { Product } from "@/components/productsData";

interface Category {
  id: string;
  name: string;
  sort_order: number;
}

interface DbProduct {
  id: string;
  name: string;
  image_url: string | null;
  price: string;
  original_price: string | null;
  category: string;
  badges: string[] | null;
  description: string | null;
  features: string[] | null;
  duration_options: string[] | null;
}

const Index = () => {
  useSiteSettings();
  usePixelPageTracking();
  usePixelInternalClickTracking();
  const [categories, setCategories] = useState<Category[]>([]);
  const [products, setProducts] = useState<Product[]>([]);

  useEffect(() => {
    const fetchData = async () => {
      const [catRes, prodRes] = await Promise.all([
        supabase.from("categories").select("*").eq("is_active", true).order("sort_order"),
        supabase.from("products").select("*"),
      ]);

      if (catRes.data) setCategories(catRes.data);
      if (prodRes.data) {
        const mapped: Product[] = prodRes.data.map((p: DbProduct) => ({
          id: p.id as any,
          name: p.name,
          slug: (p as any).slug || p.id,
          image: p.image_url || "/placeholder.svg",
          price: p.price,
          originalPrice: p.original_price || undefined,
          resellerPrice: (p as any).reseller_price || undefined,
          resellerPrices: (p as any).reseller_prices || undefined,
          category: p.category,
          badges: (p.badges || []) as ("sale" | "limited")[],
          description: p.description || undefined,
          features: p.features || undefined,
          duration: p.duration_options || undefined,
        }));
        setProducts(mapped);
      }
    };
    fetchData();
  }, []);

  const getProductsByCategory = (categoryName: string) =>
    products.filter((p) => p.category.split(",").map(s => s.trim()).includes(categoryName.trim()));

  return (
    <div className="min-h-screen bg-background">
      <TopBar />
      <Header />
      <HeroBanner />

      <div id="products" className="bg-background">
        {categories.map((cat) => {
          const catProducts = getProductsByCategory(cat.name);
          if (catProducts.length === 0) return null;
          return (
            <ProductSection
              key={cat.id}
              title={cat.name}
              id={cat.name.toLowerCase().replace(/\s+/g, "-")}
              products={catProducts}
            />
          );
        })}
      </div>

      <PartnersSection />
      <ShopFooter />
      <BottomBar />
      <AppInstallBanner />
      <PopupModal />
    </div>
  );
};

export default Index;
