import { useState, useEffect, useCallback } from "react";
import { Download, CheckCircle, Smartphone, Share, MoreVertical, Zap, Bell, Globe } from "lucide-react";
import { motion } from "framer-motion";
import { supabase } from "@/integrations/supabase/client";
import Header from "@/components/Header";
import TopBar from "@/components/TopBar";
import ShopFooter from "@/components/ShopFooter";
import BottomBar from "@/components/BottomBar";

interface BeforeInstallPromptEvent extends Event {
  prompt: () => Promise<void>;
  userChoice: Promise<{ outcome: "accepted" | "dismissed" }>;
}

const Install = () => {
  const [deferredPrompt, setDeferredPrompt] = useState<BeforeInstallPromptEvent | null>(null);
  const [isInstalled, setIsInstalled] = useState(false);
  const [isIOS, setIsIOS] = useState(false);
  const [installing, setInstalling] = useState(false);
  const [pwaIcon, setPwaIcon] = useState<string | null>(null);

  useEffect(() => {
    if (window.matchMedia("(display-mode: standalone)").matches) {
      setIsInstalled(true);
      return;
    }

    const ua = navigator.userAgent;
    setIsIOS(/iPad|iPhone|iPod/.test(ua) && !(window as any).MSStream);

    const handler = (e: Event) => {
      e.preventDefault();
      setDeferredPrompt(e as BeforeInstallPromptEvent);
    };

    window.addEventListener("beforeinstallprompt", handler);
    window.addEventListener("appinstalled", () => setIsInstalled(true));

    // Fetch PWA icon from settings
    supabase.from("site_settings").select("value").eq("key", "pwa_icon_url").maybeSingle().then(({ data }) => {
      if (data?.value) setPwaIcon(data.value);
    });

    return () => window.removeEventListener("beforeinstallprompt", handler);
  }, []);

  const handleInstall = useCallback(async () => {
    if (!deferredPrompt) return;
    setInstalling(true);
    try {
      await deferredPrompt.prompt();
      const { outcome } = await deferredPrompt.userChoice;
      if (outcome === "accepted") setIsInstalled(true);
    } finally {
      setDeferredPrompt(null);
      setInstalling(false);
    }
  }, [deferredPrompt]);

  const features = [
    { icon: <Zap className="w-6 h-6 text-primary" />, label: "দ্রুত লোড", desc: "ইন্সট্যান্ট ওপেন" },
    { icon: <Smartphone className="w-6 h-6 text-primary" />, label: "অ্যাপ ফিল", desc: "নেটিভ এক্সপেরিয়েন্স" },
    { icon: <Bell className="w-6 h-6 text-primary" />, label: "নোটিফিকেশন", desc: "আপডেট পান" },
  ];

  return (
    <div className="min-h-screen bg-background">
      <TopBar />
      <Header />

      <div className="max-w-lg mx-auto px-4 py-8 pb-24">
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          className="space-y-6"
        >
          {/* Hero */}
          <div className="text-center space-y-4">
            <motion.div
              initial={{ scale: 0.8 }}
              animate={{ scale: 1 }}
              transition={{ type: "spring", stiffness: 200 }}
              className="flex justify-center"
            >
              {pwaIcon ? (
                <img src={pwaIcon} alt="App Icon" className="w-24 h-24 rounded-3xl shadow-xl shadow-primary/20 object-cover" />
              ) : (
                <div className="w-24 h-24 rounded-3xl bg-gradient-to-br from-primary to-primary/60 flex items-center justify-center shadow-xl shadow-primary/20">
                  <Smartphone className="w-12 h-12 text-primary-foreground" />
                </div>
              )}
            </motion.div>

            <div>
              <h1 className="text-2xl font-bold text-foreground">TZSS App ইনস্টল করুন</h1>
              <p className="text-muted-foreground text-sm mt-2">
                আরও ভালো অভিজ্ঞতার জন্য আমাদের অ্যাপটি আপনার ফোনে ইনস্টল করুন
              </p>
            </div>
          </div>

          {/* Status / Action */}
          {isInstalled ? (
            <motion.div
              initial={{ scale: 0.95, opacity: 0 }}
              animate={{ scale: 1, opacity: 1 }}
              className="bg-green-500/10 border border-green-500/20 rounded-2xl p-6 text-center space-y-3"
            >
              <CheckCircle className="w-14 h-14 text-green-500 mx-auto" />
              <p className="text-green-600 font-bold text-lg">অ্যাপ ইতিমধ্যে ইনস্টল আছে! ✅</p>
              <p className="text-muted-foreground text-sm">আপনার হোম স্ক্রীন থেকে TZSS অ্যাপ খুলুন।</p>
            </motion.div>
          ) : (
            <div className="space-y-4">
              {/* Main Install Button - always visible */}
              <motion.button
                whileTap={{ scale: 0.97 }}
                onClick={deferredPrompt ? handleInstall : undefined}
                disabled={installing}
                className={`w-full py-4 rounded-2xl font-bold text-lg flex items-center justify-center gap-3 shadow-lg shadow-primary/20 transition-all ${
                  deferredPrompt
                    ? "bg-primary text-primary-foreground hover:opacity-90 cursor-pointer"
                    : "bg-primary/80 text-primary-foreground cursor-default"
                }`}
              >
                <Download className="w-6 h-6" />
                {installing ? "ইনস্টল হচ্ছে..." : "এখনই ইনস্টল করুন"}
              </motion.button>

              {!deferredPrompt && (
                <p className="text-center text-xs text-muted-foreground">
                  👆 বাটন কাজ না করলে নিচের ম্যানুয়াল পদ্ধতি অনুসরণ করুন
                </p>
              )}

              {/* Manual Instructions */}
              {isIOS ? (
                <div className="bg-muted/50 border border-border rounded-2xl p-5 space-y-4">
                  <p className="font-semibold text-foreground text-center flex items-center justify-center gap-2">
                    <Globe className="w-4 h-4" /> iOS এ ইনস্টল করুন
                  </p>
                  <div className="space-y-3">
                    {[
                      <>Safari ব্রাউজারে নিচের <Share className="w-4 h-4 inline text-primary mx-1" /> (Share) বাটনে ট্যাপ করুন</>,
                      <>"Add to Home Screen" অপশনে ট্যাপ করুন</>,
                      <>"Add" বাটনে ট্যাপ করে ইনস্টল সম্পন্ন করুন</>,
                    ].map((text, i) => (
                      <div key={i} className="flex items-start gap-3">
                        <div className="w-7 h-7 rounded-full bg-primary/10 flex items-center justify-center flex-shrink-0 mt-0.5">
                          <span className="text-primary font-bold text-xs">{i + 1}</span>
                        </div>
                        <p className="text-sm text-muted-foreground">{text}</p>
                      </div>
                    ))}
                  </div>
                </div>
              ) : (
                <div className="bg-muted/50 border border-border rounded-2xl p-5 space-y-4">
                  <p className="font-semibold text-foreground text-center flex items-center justify-center gap-2">
                    <Globe className="w-4 h-4" /> Android এ ইনস্টল করুন
                  </p>
                  <div className="space-y-3">
                    {[
                      <>Chrome ব্রাউজারে উপরের <MoreVertical className="w-4 h-4 inline text-primary mx-1" /> (Menu) বাটনে ট্যাপ করুন</>,
                      <>"Install app" বা "Add to Home screen" এ ট্যাপ করুন</>,
                      <>"Install" বাটনে ট্যাপ করে ইনস্টল সম্পন্ন করুন</>,
                    ].map((text, i) => (
                      <div key={i} className="flex items-start gap-3">
                        <div className="w-7 h-7 rounded-full bg-primary/10 flex items-center justify-center flex-shrink-0 mt-0.5">
                          <span className="text-primary font-bold text-xs">{i + 1}</span>
                        </div>
                        <p className="text-sm text-muted-foreground">{text}</p>
                      </div>
                    ))}
                  </div>
                </div>
              )}
            </div>
          )}

          {/* Features */}
          <div className="grid grid-cols-3 gap-3">
            {features.map((f) => (
              <motion.div
                key={f.label}
                whileHover={{ y: -2 }}
                className="bg-card border border-border rounded-xl p-3 text-center space-y-1.5"
              >
                <div className="flex justify-center">{f.icon}</div>
                <p className="text-xs font-semibold text-foreground">{f.label}</p>
                <p className="text-[10px] text-muted-foreground">{f.desc}</p>
              </motion.div>
            ))}
          </div>
        </motion.div>
      </div>

      <ShopFooter />
      <BottomBar />
    </div>
  );
};

export default Install;
