import { useState, useEffect } from "react";
import { useNavigate } from "react-router-dom";
import { supabase } from "@/integrations/supabase/client";
import { useAuth } from "@/contexts/AuthContext";
import { Package, Copy, CheckCircle2, ChevronDown, Key, RefreshCw, CalendarDays, CreditCard, Hash, ShoppingBag, ArrowLeft } from "lucide-react";
import { Button } from "@/components/ui/button";
import { motion, AnimatePresence } from "framer-motion";
import Header from "@/components/Header";
import TopBar from "@/components/TopBar";
import BottomBar from "@/components/BottomBar";

interface Order {
  id: string;
  product_name: string;
  amount: string;
  status: string;
  payment_status: string | null;
  payment_method: string | null;
  customer_name: string;
  customer_phone: string | null;
  transaction_id: string | null;
  created_at: string;
}

interface DeliveredKey {
  id: string;
  key_value: string;
  product_name: string;
  sold_at: string | null;
  order_id: string;
}

const Orders = () => {
  const { user, loading: authLoading } = useAuth();
  const navigate = useNavigate();
  const [orders, setOrders] = useState<Order[]>([]);
  const [deliveredKeys, setDeliveredKeys] = useState<DeliveredKey[]>([]);
  const [loading, setLoading] = useState(true);
  const [expandedOrder, setExpandedOrder] = useState<string | null>(null);
  const [copiedKey, setCopiedKey] = useState<string | null>(null);
  const [userEmail, setUserEmail] = useState<string | null>(null);

  useEffect(() => {
    if (!authLoading && !user) navigate("/auth");
  }, [user, authLoading, navigate]);

  useEffect(() => {
    if (!user) return;
    const fetchOrders = async () => {
      // Fetch profile email
      const { data: profileData } = await supabase
        .from("profiles")
        .select("email")
        .eq("user_id", user.id)
        .single();
      if (profileData?.email) setUserEmail(profileData.email);

      const { data: ordersData } = await supabase
        .from("orders")
        .select("*")
        .eq("user_id", user.id)
        .order("created_at", { ascending: false });

      if (ordersData) {
        setOrders(ordersData);
        const orderIds = ordersData.map((o: any) => o.id);
        if (orderIds.length > 0) {
          const { data: keysData } = await supabase
            .from("product_keys")
            .select("id, key_value, sold_at, order_id")
            .in("order_id", orderIds);
          if (keysData) {
            setDeliveredKeys(
              keysData.map((k: any) => {
                const order = ordersData.find((o: any) => o.id === k.order_id);
                return {
                  id: k.id,
                  key_value: k.key_value,
                  product_name: order?.product_name || "Unknown",
                  sold_at: k.sold_at,
                  order_id: k.order_id,
                };
              })
            );
          }
        }
      }
      setLoading(false);
    };
    fetchOrders();
  }, [user]);

  const statusColor = (status: string) => {
    switch (status) {
      case "completed": return "bg-primary/15 text-primary border border-primary/20";
      case "pending": return "bg-accent/15 text-accent-foreground border border-accent/20";
      case "cancelled": return "bg-destructive/10 text-destructive border border-destructive/20";
      default: return "bg-muted text-muted-foreground";
    }
  };

  const completedCount = orders.filter(o => o.status === "completed").length;
  const pendingCount = orders.filter(o => o.status === "pending" || o.status === "processing").length;

  if (authLoading || loading) {
    return (
      <div className="min-h-screen flex flex-col items-center justify-center bg-background gap-3">
        <motion.div
          animate={{ rotate: 360 }}
          transition={{ duration: 1, repeat: Infinity, ease: "linear" }}
          className="h-10 w-10 border-4 border-primary border-t-transparent rounded-full"
        />
        <p className="text-sm text-muted-foreground">লোড হচ্ছে...</p>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-b from-primary/10 via-primary/5 to-background">
      <TopBar />
      <Header />
      <div className="pb-24 sm:pb-4">
        {/* Hero Header */}
        <div className="relative">
          <div className="absolute inset-0 bg-gradient-to-br from-primary/15 via-primary/5 to-transparent" />
          <div className="absolute inset-0 bg-[radial-gradient(circle_at_30%_40%,hsl(var(--primary)/0.12),transparent_50%)]" />
          <motion.div
            animate={{ y: [-6, 6, -6], x: [-3, 3, -3] }}
            transition={{ duration: 7, repeat: Infinity, ease: "easeInOut" }}
            className="absolute top-8 right-[12%] w-24 h-24 rounded-full bg-primary/8 blur-2xl"
          />

          <div className="container mx-auto px-3 sm:px-4 pt-6 sm:pt-10 pb-14 sm:pb-16 max-w-2xl relative z-10">
            <motion.button
              initial={{ opacity: 0, x: -10 }}
              animate={{ opacity: 1, x: 0 }}
              onClick={() => navigate("/")}
              className="hidden sm:flex items-center gap-1.5 text-sm text-muted-foreground hover:text-foreground mb-6 transition-colors group"
            >
              <ArrowLeft className="h-4 w-4 group-hover:-translate-x-1 transition-transform" /> হোমে ফিরে যান
            </motion.button>

            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5 }}
              className="flex flex-col items-center text-center"
            >
              <div className="w-16 h-16 rounded-2xl bg-primary/10 flex items-center justify-center mb-4 border border-primary/20">
                <Package className="h-8 w-8 text-primary" />
              </div>
              <h1 className="font-heading text-2xl sm:text-3xl font-bold text-foreground tracking-tight">
                আমার অর্ডার সমূহ
              </h1>
              <p className="text-sm text-muted-foreground mt-1.5">আপনার সকল অর্ডারের তালিকা</p>
            </motion.div>
          </div>
        </div>

        <div className="container mx-auto px-3 sm:px-4 max-w-2xl -mt-6 relative z-20 space-y-4">
          {/* Quick Stats */}
          <motion.div
            initial={{ opacity: 0, y: 10 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.2 }}
            className="grid grid-cols-3 gap-3"
          >
            {[
              { icon: <ShoppingBag className="h-5 w-5" />, value: orders.length, label: "মোট অর্ডার" },
              { icon: <CheckCircle2 className="h-5 w-5" />, value: completedCount, label: "সম্পন্ন" },
              { icon: <Key className="h-5 w-5" />, value: deliveredKeys.length, label: "লাইসেন্স" },
            ].map((stat, i) => (
              <motion.div
                key={i}
                initial={{ opacity: 0, scale: 0.9 }}
                animate={{ opacity: 1, scale: 1 }}
                transition={{ delay: 0.3 + i * 0.1 }}
                className="bg-card border border-border rounded-2xl p-4 text-center relative overflow-hidden group hover:border-primary/30 transition-colors"
              >
                <div className="absolute inset-0 bg-gradient-to-br from-primary/5 to-transparent opacity-0 group-hover:opacity-100 transition-opacity" />
                <div className="text-primary/60 mb-1.5 flex justify-center relative z-10">{stat.icon}</div>
                <p className="text-3xl font-bold text-foreground relative z-10">{stat.value}</p>
                <p className="text-[10px] text-muted-foreground font-medium mt-1 relative z-10">{stat.label}</p>
              </motion.div>
            ))}
          </motion.div>

          {/* Orders List */}
          <motion.div
            initial={{ opacity: 0, y: 15 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.4 }}
          >
            {orders.length === 0 ? (
              <div className="bg-card border border-border rounded-2xl shadow-sm overflow-hidden">
                <div className="text-center py-20 px-4">
                  <motion.div
                    initial={{ scale: 0.8, opacity: 0 }}
                    animate={{ scale: 1, opacity: 1 }}
                    transition={{ type: "spring", stiffness: 200 }}
                  >
                    <div className="w-20 h-20 mx-auto mb-5 rounded-2xl bg-gradient-to-br from-muted to-muted/50 flex items-center justify-center">
                      <Package className="h-10 w-10 text-muted-foreground/30" />
                    </div>
                    <p className="font-heading font-bold text-foreground text-lg">এখনো কোনো অর্ডার নেই</p>
                    <p className="text-sm text-muted-foreground mt-1.5">প্রোডাক্ট কিনলে এখানে দেখা যাবে</p>
                    <Button onClick={() => navigate("/shop")} className="mt-5 rounded-xl shadow-md" size="sm">
                      <ShoppingBag className="h-4 w-4 mr-1.5" /> শপে যান
                    </Button>
                  </motion.div>
                </div>
              </div>
            ) : (
              <div className="bg-card border border-border rounded-2xl shadow-sm overflow-hidden">
                <div className="relative px-5 py-4 border-b border-border overflow-hidden">
                  <div className="absolute inset-0 bg-gradient-to-r from-primary/8 via-primary/3 to-transparent" />
                  <h2 className="font-heading text-lg font-bold text-foreground flex items-center gap-2 relative z-10">
                    <div className="w-8 h-8 rounded-xl bg-primary/10 flex items-center justify-center">
                      <Package className="h-4 w-4 text-primary" />
                    </div>
                    অর্ডার হিস্ট্রি
                  </h2>
                  <p className="text-xs text-muted-foreground mt-1 relative z-10 ml-10">
                    মোট {orders.length}টি অর্ডার • {completedCount}টি সম্পন্ন • {pendingCount}টি পেন্ডিং
                  </p>
                </div>

                <div className="divide-y divide-border">
                  {orders.map((order, idx) => {
                    const orderKeys = deliveredKeys.filter(k => k.order_id === order.id);
                    const isExpanded = expandedOrder === order.id;
                    return (
                      <motion.div
                        key={order.id}
                        initial={{ opacity: 0 }}
                        animate={{ opacity: 1 }}
                        transition={{ delay: idx * 0.04 }}
                      >
                        <button
                          className="w-full flex items-center justify-between p-4 sm:p-5 hover:bg-muted/20 transition-colors text-left"
                          onClick={() => setExpandedOrder(isExpanded ? null : order.id)}
                        >
                          <div className="flex items-start gap-3 flex-1 min-w-0">
                            <div className={`w-10 h-10 rounded-xl flex items-center justify-center flex-shrink-0 ${
                              order.status === "completed" ? "bg-primary/10" : "bg-muted"
                            }`}>
                              <Package className={`h-5 w-5 ${order.status === "completed" ? "text-primary" : "text-muted-foreground"}`} />
                            </div>
                            <div className="flex-1 min-w-0">
                              <p className="text-sm font-semibold text-foreground truncate">{order.product_name}</p>
                              <div className="flex items-center gap-2 mt-1 flex-wrap">
                                <span className="font-mono text-[10px] text-muted-foreground">#{order.id.slice(0, 8)}</span>
                                <span className={`text-[10px] font-semibold px-2 py-0.5 rounded-full ${statusColor(order.status)}`}>
                                  {order.status === "completed" ? "✓ সম্পন্ন" : order.status === "pending" || order.status === "processing" ? "⏳ পেন্ডিং" : order.status}
                                </span>
                                {orderKeys.length > 0 && (
                                  <span className="text-[9px] text-primary font-semibold">🔑 কী আছে</span>
                                )}
                              </div>
                              <p className="text-[10px] text-muted-foreground mt-1 flex items-center gap-1">
                                <CalendarDays className="h-3 w-3" />
                                {new Date(order.created_at).toLocaleDateString("bn-BD", { year: "numeric", month: "short", day: "numeric" })}
                              </p>
                            </div>
                          </div>
                          <div className="flex items-center gap-3 ml-3 flex-shrink-0">
                            <span className="font-bold text-foreground text-sm">৳{order.amount}</span>
                            <motion.div animate={{ rotate: isExpanded ? 180 : 0 }} transition={{ duration: 0.2 }}>
                              <ChevronDown className="h-4 w-4 text-muted-foreground" />
                            </motion.div>
                          </div>
                        </button>

                        <AnimatePresence>
                          {isExpanded && (
                            <motion.div
                              initial={{ height: 0, opacity: 0 }}
                              animate={{ height: "auto", opacity: 1 }}
                              exit={{ height: 0, opacity: 0 }}
                              transition={{ duration: 0.25, ease: "easeInOut" }}
                              className="overflow-hidden"
                            >
                              <div className="border-t border-border p-4 sm:p-5 space-y-3 bg-muted/10">
                                {/* Payment, Total & Date */}
                                <div className="grid grid-cols-3 gap-3">
                                  <div className="bg-card rounded-xl p-3.5 border border-border">
                                    <div className="flex items-center gap-1.5 mb-1">
                                      <CreditCard className="h-3 w-3 text-primary" />
                                      <p className="text-[10px] text-muted-foreground font-semibold uppercase tracking-wider">পেমেন্ট</p>
                                    </div>
                                    <p className="text-xs font-bold text-foreground">{order.payment_method || "N/A"}</p>
                                  </div>
                                  <div className="bg-card rounded-xl p-3.5 border border-border">
                                    <div className="flex items-center gap-1.5 mb-1">
                                      <Hash className="h-3 w-3 text-primary" />
                                      <p className="text-[10px] text-muted-foreground font-semibold uppercase tracking-wider">মোট</p>
                                    </div>
                                    <p className="text-xs font-bold text-foreground">৳{order.amount}</p>
                                  </div>
                                  <div className="bg-card rounded-xl p-3.5 border border-border">
                                    <div className="flex items-center gap-1.5 mb-1">
                                      <CalendarDays className="h-3 w-3 text-primary" />
                                      <p className="text-[10px] text-muted-foreground font-semibold uppercase tracking-wider">তারিখ</p>
                                    </div>
                                    <p className="text-xs font-bold text-foreground">
                                      {new Date(order.created_at).toLocaleDateString("bn-BD", { day: "numeric", month: "short" })}
                                    </p>
                                    <p className="text-[10px] text-muted-foreground">
                                      {new Date(order.created_at).toLocaleTimeString("bn-BD", { hour: "2-digit", minute: "2-digit" })}
                                    </p>
                                  </div>
                                </div>

                                {/* Customer Info */}
                                <div className="bg-card rounded-xl p-3.5 border border-border">
                                  <p className="text-[10px] text-muted-foreground font-semibold uppercase tracking-wider mb-2">কাস্টমার তথ্য</p>
                                  <p className="text-sm text-foreground font-medium">{order.customer_name}</p>
                                  {order.customer_phone && <p className="text-xs text-muted-foreground mt-0.5">📞 {order.customer_phone}</p>}
                                  {userEmail && <p className="text-xs text-muted-foreground mt-0.5">✉️ {userEmail}</p>}
                                  {order.transaction_id && <p className="text-xs text-muted-foreground font-mono mt-0.5">TXN: {order.transaction_id}</p>}
                                </div>

                                {/* Keys */}
                                {orderKeys.length > 0 && (
                                  <div className="bg-primary/5 rounded-xl p-3.5 border border-primary/15">
                                    <p className="text-[10px] text-primary font-bold uppercase tracking-wider mb-2 flex items-center gap-1.5">
                                      <Key className="h-3 w-3" /> লাইসেন্স কী
                                    </p>
                                    <div className="space-y-2">
                                      {orderKeys.map((dk) => (
                                        <div key={dk.id} className="bg-card rounded-lg p-3 flex items-center gap-2 border border-border">
                                          <code className="text-xs font-mono flex-1 break-all text-foreground">{dk.key_value}</code>
                                          <motion.button
                                            whileTap={{ scale: 0.9 }}
                                            onClick={(e) => {
                                              e.stopPropagation();
                                              navigator.clipboard.writeText(dk.key_value);
                                              setCopiedKey(dk.id);
                                              setTimeout(() => setCopiedKey(null), 2000);
                                            }}
                                            className="flex-shrink-0 p-2 rounded-lg hover:bg-muted transition-colors"
                                          >
                                            {copiedKey === dk.id ? <CheckCircle2 className="h-4 w-4 text-primary" /> : <Copy className="h-4 w-4 text-muted-foreground" />}
                                          </motion.button>
                                        </div>
                                      ))}
                                    </div>
                                  </div>
                                )}

                                {/* Reorder */}
                                <Button
                                  size="sm"
                                  className="rounded-xl text-xs h-9 shadow-sm"
                                  onClick={(e) => {
                                    e.stopPropagation();
                                    supabase
                                      .from("products")
                                      .select("slug")
                                      .eq("name", order.product_name)
                                      .single()
                                      .then(({ data }) => {
                                        if (data?.slug) {
                                          navigate(`/product/${data.slug}`);
                                        } else {
                                          navigate("/shop");
                                        }
                                      });
                                  }}
                                >
                                  <RefreshCw className="h-3.5 w-3.5 mr-1" /> আবার অর্ডার করুন
                                </Button>

                                <p className="text-[8px] text-muted-foreground/40 font-mono break-all pt-1">ID: {order.id}</p>
                              </div>
                            </motion.div>
                          )}
                        </AnimatePresence>
                      </motion.div>
                    );
                  })}
                </div>
              </div>
            )}
          </motion.div>
        </div>
      </div>
    </div>
  );
};

export default Orders;
