import { useState, useEffect } from "react";
import { trackPixelEvent, usePixelPageTracking } from "@/hooks/usePixel";
import { useParams, Link, useNavigate } from "react-router-dom";
import { ShoppingCart, ChevronRight, Check, Shield, Truck, RefreshCw, Zap, Star, MessageCircle, Package, Eye, Clock, ThumbsUp, Copy, Send, HelpCircle, ChevronDown as ChevronDownIcon, Share2, Facebook, ExternalLink } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { toast } from "@/hooks/use-toast";
import { useCart } from "@/contexts/CartContext";
import { useAuth } from "@/contexts/AuthContext";
import TopBar from "@/components/TopBar";
import Header from "@/components/Header";
import ShopFooter from "@/components/ShopFooter";
import BottomBar from "@/components/BottomBar";
import ProductSection from "@/components/ProductSection";
import { supabase } from "@/integrations/supabase/client";
import type { Product } from "@/components/productsData";
import { parseDurationOption, getPriceRangeFromDurations } from "@/lib/priceUtils";
import { useSiteSettings } from "@/hooks/useSiteSettings";
import { motion } from "framer-motion";
import { Textarea } from "@/components/ui/textarea";

const mapDbProduct = (p: any): Product => ({
  id: p.id,
  name: p.name,
  slug: p.slug || p.id,
  image: p.image_url || "/placeholder.svg",
  price: p.price,
  originalPrice: p.original_price || undefined,
  resellerPrice: p.reseller_price || undefined,
  resellerPrices: p.reseller_prices || undefined,
  category: p.category,
  badges: (p.badges || []) as ("sale" | "limited")[],
  description: p.description || undefined,
  longDescription: p.long_description || undefined,
  features: Array.isArray(p.features)
    ? p.features.map((f: string) => f?.trim()).filter(Boolean)
    : typeof p.features === "string"
      ? p.features.split(/\r?\n|,/).map((f: string) => f.trim()).filter(Boolean)
      : undefined,
  duration: p.duration_options || undefined,
  customFields: p.custom_fields || undefined,
});

const ProductDetail = () => {
  const { id } = useParams<{ id: string }>();
  const [product, setProduct] = useState<Product | null>(null);
  const [relatedProducts, setRelatedProducts] = useState<Product[]>([]);
  const [loading, setLoading] = useState(true);
  const [selectedDuration, setSelectedDuration] = useState<string | null>(null);
  const [quantity, setQuantity] = useState(1);
  const [customFieldValues, setCustomFieldValues] = useState<Record<string, string>>({});
  const [activeTab, setActiveTab] = useState<"description" | "features" | "info">("description");
  const [reviews, setReviews] = useState<any[]>([]);
  const [reviewRating, setReviewRating] = useState(5);
  const [reviewComment, setReviewComment] = useState("");
  const [submittingReview, setSubmittingReview] = useState(false);
  const [faqs, setFaqs] = useState<{ id: string; question: string; answer: string }[]>([]);
  const [openFaqId, setOpenFaqId] = useState<string | null>(null);
  const { addToCart } = useCart();
  const { isReseller, user } = useAuth();
  const { settings } = useSiteSettings();
  usePixelPageTracking();

  const hasDurationPrices = product?.duration?.some(
    (d) => parseDurationOption(d).price !== null
  );

  const displayPrice = (() => {
    if (!product) return "";
    if (selectedDuration) {
      const parsed = parseDurationOption(selectedDuration);
      return parsed.price || product.price;
    }
    if (hasDurationPrices && product.duration) {
      return getPriceRangeFromDurations(product.duration) || product.price;
    }
    return product.price;
  })();
  const navigate = useNavigate();

  useEffect(() => {
    const fetchProduct = async () => {
      if (!id) return;
      setLoading(true);
      let { data } = await supabase.from("products").select("*").eq("slug", id).maybeSingle();
      if (!data) {
        const res = await supabase.from("products").select("*").eq("id", id).maybeSingle();
        data = res.data;
      }
      if (data) {
        const mapped = mapDbProduct(data);
        setProduct(mapped);
        trackPixelEvent("ViewContent", { content_name: mapped.name, content_category: mapped.category, value: parseFloat(mapped.price.replace(/[^\d.]/g, "")) || 0, currency: "BDT" });
        const { data: related } = await supabase
          .from("products")
          .select("*")
          .eq("category", data.category)
          .neq("id", data.id)
          .limit(4);
        if (related) setRelatedProducts(related.map(mapDbProduct));

        // Fetch approved reviews
        const { data: reviewsData } = await supabase
          .from("product_reviews")
          .select("*")
          .eq("product_id", data.id)
          .eq("is_approved", true)
          .order("created_at", { ascending: false });
        if (reviewsData) setReviews(reviewsData);

        // Fetch FAQs
        const { data: faqsData } = await supabase
          .from("product_faqs")
          .select("*")
          .eq("product_id", data.id)
          .order("sort_order");
        if (faqsData) setFaqs(faqsData);
      }
      setLoading(false);
    };
    fetchProduct();
  }, [id]);

  if (loading) {
    return (
      <div className="min-h-screen bg-background">
        <TopBar />
        <Header />
        <div className="container mx-auto px-4 py-20 flex flex-col items-center gap-3">
          <div className="h-10 w-10 border-4 border-primary border-t-transparent rounded-full animate-spin" />
          <p className="text-muted-foreground text-sm">লোড হচ্ছে...</p>
        </div>
        <ShopFooter />
      </div>
    );
  }

  if (!product) {
    return (
      <div className="min-h-screen bg-background">
        <TopBar />
        <Header />
        <div className="container mx-auto px-4 py-20 text-center">
          <h1 className="font-heading text-2xl font-bold text-foreground mb-4">Product Not Found</h1>
          <Link to="/" className="text-primary hover:underline">← Back to Home</Link>
        </div>
        <ShopFooter />
      </div>
    );
  }

  const validateSelection = () => {
    if (product!.duration && product!.duration.length > 0 && !selectedDuration) {
      toast({ title: "অপশন সিলেক্ট করুন", description: "দয়া করে একটি ডিউরেশন/অপশন সিলেক্ট করুন।", variant: "destructive" });
      return false;
    }
    if (product!.customFields && product!.customFields.length > 0) {
      for (const field of product!.customFields) {
        if (field.required && !customFieldValues[field.label]?.trim()) {
          toast({ title: `${field.label} পূরণ করুন`, description: `দয়া করে ${field.label} ফিল্ডটি পূরণ করুন।`, variant: "destructive" });
          return false;
        }
      }
    }
    return true;
  };

  const handleAddToCart = () => {
    if (!user) {
      toast({ title: "লগইন করুন", description: "কার্টে যোগ করতে আগে লগইন করতে হবে।", variant: "destructive" });
      navigate("/auth", { state: { from: `/product/${id}` } });
      return;
    }
    if (!validateSelection()) return;
    const cfv = product!.customFields && product!.customFields.length > 0 ? customFieldValues : undefined;
    addToCart(product!, selectedDuration, quantity, cfv);
    trackPixelEvent("AddToCart", { content_name: product!.name, value: parseFloat(displayPrice.replace(/[^\d.]/g, "")) || 0, currency: "BDT" });
    toast({ title: "কার্টে যোগ হয়েছে! ✅", description: `${product!.name}${selectedDuration ? ` (${selectedDuration})` : ""} × ${quantity} কার্টে যোগ করা হয়েছে।` });
  };

  const handleBuyNow = () => {
    if (!user) {
      toast({ title: "লগইন করুন", description: "অর্ডার করতে আগে লগইন করতে হবে।", variant: "destructive" });
      navigate("/auth", { state: { from: `/product/${id}` } });
      return;
    }
    if (!validateSelection()) return;
    const cfv = product!.customFields && product!.customFields.length > 0 ? customFieldValues : undefined;
    addToCart(product!, selectedDuration, quantity, cfv);
    navigate("/checkout");
  };

  const handleSubmitReview = async () => {
    if (!user) {
      toast({ title: "লগইন করুন", description: "রিভিউ দিতে প্রথমে লগইন করুন।", variant: "destructive" });
      navigate("/auth");
      return;
    }
    if (!reviewComment.trim()) {
      toast({ title: "কমেন্ট লিখুন", description: "দয়া করে আপনার মতামত লিখুন।", variant: "destructive" });
      return;
    }
    setSubmittingReview(true);
    const userName = user.user_metadata?.name || user.email?.split("@")[0] || "Anonymous";
    const { error } = await supabase.from("product_reviews").insert({
      product_id: product!.id as string,
      user_id: user.id,
      user_name: userName,
      rating: reviewRating,
      comment: reviewComment.trim(),
    });
    setSubmittingReview(false);
    if (!error) {
      toast({ title: "রিভিউ সাবমিট হয়েছে! ✅", description: "অ্যাডমিন অ্যাপ্রুভ করলে এটি দেখা যাবে।" });
      setReviewComment("");
      setReviewRating(5);
    } else {
      toast({ title: "Error", description: error.message, variant: "destructive" });
    }
  };

  const tabs = [
    { key: "description" as const, label: "বিবরণ", icon: Eye },
    { key: "features" as const, label: "ফিচারস", icon: Star },
    { key: "info" as const, label: "তথ্য", icon: Package },
  ];

  return (
    <div className="min-h-screen bg-background pb-20 md:pb-0">
      <TopBar />
      <Header />

      {/* Breadcrumb */}
      <div className="bg-muted/30 border-b border-border">
        <div className="container mx-auto px-3 sm:px-4 py-2">
          <div className="flex items-center gap-1.5 text-xs text-muted-foreground overflow-x-auto">
            <Link to="/" className="hover:text-primary transition-colors shrink-0">হোম</Link>
            <ChevronRight className="h-3 w-3 shrink-0" />
            <Link to="/shop" className="hover:text-primary transition-colors shrink-0">{product.category}</Link>
            <ChevronRight className="h-3 w-3 shrink-0" />
            <span className="text-foreground font-medium truncate">{product.name}</span>
          </div>
        </div>
      </div>

      {/* Hero Product Section */}
      <section className="py-4 sm:py-8">
        <div className="container mx-auto px-3 sm:px-4">
          <div className="grid md:grid-cols-2 gap-4 sm:gap-8 lg:gap-12">
            {/* Product Image */}
            <motion.div
              initial={{ opacity: 0, x: -20 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ duration: 0.4 }}
            >
              <div className="relative bg-gradient-to-br from-primary/5 via-muted/50 to-primary/10 rounded-2xl border border-border overflow-hidden aspect-square md:sticky md:top-24">
                {/* Badges */}
                <div className="absolute top-3 left-3 flex flex-col gap-1.5 z-10">
                  {product.badges?.includes("sale") && (
                    <span className="px-2.5 py-1 text-[10px] font-bold rounded-full bg-destructive text-white">🔥 SALE</span>
                  )}
                  {product.badges?.includes("limited") && (
                    <span className="px-2.5 py-1 text-[10px] font-bold rounded-full bg-[hsl(var(--badge-limited))] text-white">⚡ LIMITED</span>
                  )}
                </div>
                {/* Category tag */}
                <span className="absolute top-3 right-3 px-2.5 py-1 text-[10px] font-semibold rounded-full bg-primary/10 text-primary backdrop-blur-sm z-10">
                  {product.category}
                </span>
                <img
                  src={product.image}
                  alt={product.name}
                  className="w-full h-full object-cover"
                />
              </div>
            </motion.div>

            {/* Product Info */}
            <motion.div
              initial={{ opacity: 0, x: 20 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ duration: 0.4, delay: 0.1 }}
              className="flex flex-col"
            >
              {/* Product name */}
              <h1 className="font-heading text-xl sm:text-2xl md:text-3xl font-bold text-foreground leading-tight mb-2">
                {product.name}
              </h1>

              {/* Rating & meta */}
              <div className="flex items-center gap-3 mb-4">
                <div className="flex items-center gap-0.5">
                  {[1, 2, 3, 4, 5].map((s) => (
                    <Star key={s} className="h-3.5 w-3.5 fill-yellow-400 text-yellow-400" />
                  ))}
                </div>
                <span className="text-xs text-muted-foreground">(4.8/5)</span>
                <span className="text-xs text-muted-foreground flex items-center gap-1">
                  <Eye className="h-3 w-3" /> ১২০+ ভিউ
                </span>
              </div>

              {/* Price block */}
              <div className="bg-gradient-to-r from-primary/5 to-transparent rounded-xl p-4 mb-4 border border-primary/10">
                <div className="flex items-baseline gap-3 flex-wrap">
                  <span className="font-heading text-2xl sm:text-3xl font-extrabold text-primary">
                    {displayPrice}
                  </span>
                  {product.originalPrice && !selectedDuration && !hasDurationPrices && (
                    <span className="text-base text-muted-foreground line-through">{product.originalPrice}</span>
                  )}
                  {product.originalPrice && !selectedDuration && !hasDurationPrices && (
                    <span className="px-2 py-0.5 text-[10px] font-bold rounded-full bg-destructive/10 text-destructive">
                      SAVE {(() => {
                        const orig = parseFloat(product.originalPrice!.replace(/[^\d.]/g, ""));
                        const curr = parseFloat(product.price.replace(/[^\d.]/g, ""));
                        if (orig && curr) return Math.round(((orig - curr) / orig) * 100);
                        return 0;
                      })()}%
                    </span>
                  )}
                </div>
                {isReseller && (
                  (() => {
                    const resellerP = selectedDuration && product.resellerPrices?.[selectedDuration]
                      ? product.resellerPrices[selectedDuration]
                      : product.resellerPrice;
                    return resellerP ? (
                      <div className="mt-2 inline-flex items-center gap-1.5 px-3 py-1 rounded-full bg-accent/10 text-accent-foreground text-xs font-semibold border border-accent/20">
                        🏪 রিসেলার প্রাইস: {resellerP}
                      </div>
                    ) : null;
                  })()
                )}
              </div>

              {/* Short description */}
              {product.description && (
                <p className="text-muted-foreground text-sm leading-relaxed mb-4 line-clamp-3">{product.description}</p>
              )}

              {/* Duration/Options */}
              {product.duration && product.duration.length > 0 && (
                <div className="mb-4">
                  <label className="block text-sm font-bold text-foreground mb-2.5">
                    🎯 {product.category === "Gift Card" ? "Amount" : "ভার্সন সিলেক্ট করুন"}
                  </label>
                  <div className="grid grid-cols-2 gap-2.5">
                    {product.duration.map((d) => {
                      const { label, price } = parseDurationOption(d);
                      const isSelected = selectedDuration === d;
                      return (
                        <button
                          key={d}
                          onClick={() => setSelectedDuration(d)}
                          className={`relative px-3 py-3.5 text-sm font-medium border-2 rounded-xl transition-all flex flex-col items-center gap-1 ${
                            isSelected
                              ? "border-primary bg-primary/10 text-foreground ring-2 ring-primary/30 shadow-md"
                              : "border-border bg-card text-foreground hover:border-primary/40 hover:bg-primary/5"
                          }`}
                        >
                          {isSelected && (
                            <span className="absolute -top-2 -right-2 h-5 w-5 bg-primary rounded-full flex items-center justify-center shadow-sm">
                              <Check className="h-3 w-3 text-primary-foreground" />
                            </span>
                          )}
                          <span className="font-semibold text-xs text-muted-foreground">{label}</span>
                          <span className={`text-base font-extrabold ${isSelected ? "text-primary" : "text-foreground"}`}>
                            {price || product.price}
                          </span>
                        </button>
                      );
                    })}
                  </div>
                </div>
              )}

              {/* Custom Fields */}
              {product.customFields && product.customFields.length > 0 && (
                <div className="mb-4 space-y-3">
                  {product.customFields.map((field) => (
                    <div key={field.label} className="space-y-1.5">
                      <label className="block text-sm font-bold text-foreground">
                        {field.label} {field.required && <span className="text-destructive">*</span>}
                        {field.helper_text && (
                          <span className="ml-2 text-xs font-normal text-muted-foreground">{field.helper_text}</span>
                        )}
                      </label>
                      <Input
                        value={customFieldValues[field.label] || ""}
                        onChange={(e) => setCustomFieldValues(prev => ({ ...prev, [field.label]: e.target.value }))}
                        placeholder={field.placeholder || `${field.label} লিখুন`}
                        className="rounded-xl border-2 focus:border-primary"
                      />
                    </div>
                  ))}
                </div>
              )}

              {/* Quantity */}
              <div className="mb-5">
                <label className="block text-sm font-bold text-foreground mb-2">পরিমাণ</label>
                <div className="flex items-center border-2 border-border rounded-xl w-fit overflow-hidden">
                  <button
                    onClick={() => setQuantity(Math.max(1, quantity - 1))}
                    className="px-4 py-2.5 text-lg font-bold text-foreground hover:bg-muted transition-colors"
                  >−</button>
                  <span className="px-6 py-2.5 text-sm font-bold text-foreground border-x-2 border-border min-w-[50px] text-center bg-muted/30">
                    {quantity}
                  </span>
                  <button
                    onClick={() => setQuantity(quantity + 1)}
                    className="px-4 py-2.5 text-lg font-bold text-foreground hover:bg-muted transition-colors"
                  >+</button>
                </div>
              </div>

              {/* Action Buttons */}
              <div className="flex flex-col gap-2.5">
                <Button
                  onClick={handleBuyNow}
                  size="lg"
                  className="w-full rounded-xl bg-primary text-primary-foreground hover:opacity-90 h-12 sm:h-14 text-base sm:text-lg font-bold gap-2 shadow-none"
                >
                  <Zap className="h-5 w-5" /> এখনই কিনুন
                </Button>
                <Button
                  onClick={handleAddToCart}
                  size="lg"
                  variant="outline"
                  className="w-full rounded-xl h-12 sm:h-14 text-base sm:text-lg font-bold gap-2 border-2 border-primary text-primary hover:bg-primary/5 shadow-none"
                >
                  <ShoppingCart className="h-5 w-5" /> কার্টে যোগ করুন
                </Button>
              </div>

              {/* Trust Badge Banner */}
              {(settings.trust_badge_url || true) && (
                <div className="mt-4 rounded-xl overflow-hidden border border-border">
                  <img 
                    src={settings.trust_badge_url || "/images/trust-badge.webp"} 
                    alt="Secure & Verified - Buy with Confidence" 
                    className="w-full h-auto object-contain"
                  />
                </div>
              )}

              {/* Trust badges */}
              <div className="mt-4 grid grid-cols-3 gap-2">
                {[
                  { icon: Shield, label: "Secure Payment", color: "text-emerald-500" },
                  { icon: Zap, label: "Instant Delivery", color: "text-amber-500" },
                  { icon: RefreshCw, label: "24/7 Support", color: "text-blue-500" },
                ].map(({ icon: Icon, label, color }) => (
                  <div key={label} className="flex flex-col items-center gap-1 p-2.5 bg-muted/40 rounded-xl text-center border border-border/50">
                    <Icon className={`h-5 w-5 ${color}`} />
                    <span className="text-[10px] font-semibold text-muted-foreground leading-tight">{label}</span>
                  </div>
                ))}
              </div>

              {/* Contact buttons */}
              <div className="mt-4">
                <label className="block text-sm font-bold text-foreground mb-2">📞 যোগাযোগ করুন</label>
                <div className="flex items-center gap-2">
                  <button
                    onClick={() => {
                      const text = `হ্যালো, আমি "${product.name}" প্রোডাক্টটি সম্পর্কে জানতে চাই।`;
                      window.open(`https://wa.me/${settings.whatsapp_number || "8801700000000"}?text=${encodeURIComponent(text)}`, "_blank");
                    }}
                    className="flex items-center gap-1.5 px-3 py-2 rounded-xl text-xs font-semibold bg-[#25D366]/10 text-[#25D366] hover:bg-[#25D366]/20 transition-colors border border-[#25D366]/20"
                  >
                    <MessageCircle className="h-3.5 w-3.5" /> WhatsApp
                  </button>
                  <button
                    onClick={() => {
                      window.open(settings.facebook_url || "https://m.me/", "_blank");
                    }}
                    className="flex items-center gap-1.5 px-3 py-2 rounded-xl text-xs font-semibold bg-[#1877F2]/10 text-[#1877F2] hover:bg-[#1877F2]/20 transition-colors border border-[#1877F2]/20"
                  >
                    <Send className="h-3.5 w-3.5" /> Messenger
                  </button>
                  <button
                    onClick={() => {
                      navigator.clipboard.writeText(window.location.href);
                      toast({ title: "✅ লিঙ্ক কপি হয়েছে!" });
                    }}
                    className="flex items-center gap-1.5 px-3 py-2 rounded-xl text-xs font-semibold bg-muted text-foreground hover:bg-muted/80 transition-colors border border-border"
                  >
                    <Copy className="h-3.5 w-3.5" /> লিঙ্ক কপি
                  </button>
                </div>
              </div>

              {/* Payment Methods */}
              <div className="mt-4 rounded-xl border border-border/50 overflow-hidden">
                <img src={settings.payment_banner_url || "/payments/payment-methods.png"} alt="Payment Methods - bKash, Nagad, Visa, Mastercard & more" className="w-full h-auto" />
              </div>
            </motion.div>
          </div>
        </div>
      </section>

      {/* Tabbed Content Section */}
      <section className="py-4 sm:py-8">
        <div className="container mx-auto px-3 sm:px-4">
          {/* Tab Headers */}
          <div className="flex border-b-2 border-border mb-5 overflow-x-auto">
            {tabs.map((tab) => (
              <button
                key={tab.key}
                onClick={() => setActiveTab(tab.key)}
                className={`flex items-center gap-1.5 px-4 py-3 text-sm font-semibold whitespace-nowrap transition-all border-b-2 -mb-[2px] ${
                  activeTab === tab.key
                    ? "border-primary text-primary"
                    : "border-transparent text-muted-foreground hover:text-foreground"
                }`}
              >
                <tab.icon className="h-4 w-4" />
                {tab.label}
              </button>
            ))}
          </div>

          {/* Tab Content */}
          <motion.div
            key={activeTab}
            initial={{ opacity: 0, y: 10 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.25 }}
          >
            {activeTab === "description" && (
              <div className="prose prose-sm max-w-none">
                <div className="bg-card rounded-2xl border border-border p-4 sm:p-6">
                  <h3 className="font-heading text-lg font-bold text-foreground mb-3 flex items-center gap-2">
                    📝 বিস্তারিত বিবরণ
                  </h3>
                  {product.description && (
                    <p className="text-foreground text-sm font-medium mb-3">{product.description}</p>
                  )}
                  <div className="text-muted-foreground text-sm leading-relaxed whitespace-pre-line">
                    {product.longDescription || (!product.description ? "এই প্রোডাক্ট সম্পর্কে কোনো বিস্তারিত বিবরণ এখনো যোগ করা হয়নি।" : "")}
                  </div>
                </div>
              </div>
            )}

            {activeTab === "features" && (
              <div className="bg-card rounded-2xl border border-border p-4 sm:p-6">
                <h3 className="font-heading text-lg font-bold text-foreground mb-4 flex items-center gap-2">
                  ⭐ ফিচারস ও সুবিধা
                </h3>
                {product.features && product.features.length > 0 ? (
                  <div className="grid sm:grid-cols-2 gap-3">
                    {product.features.map((f, i) => (
                      <div key={i} className="flex items-start gap-3 p-3 bg-muted/30 rounded-xl border border-border/50">
                        <div className="h-6 w-6 rounded-full bg-primary/10 flex items-center justify-center flex-shrink-0 mt-0.5">
                          <Check className="h-3.5 w-3.5 text-primary" />
                        </div>
                        <span className="text-sm text-foreground">{f}</span>
                      </div>
                    ))}
                  </div>
                ) : (
                  <p className="text-sm text-muted-foreground">কোনো ফিচার যোগ করা হয়নি।</p>
                )}
              </div>
            )}

            {activeTab === "info" && (
              <div className="bg-card rounded-2xl border border-border p-4 sm:p-6">
                <h3 className="font-heading text-lg font-bold text-foreground mb-4 flex items-center gap-2">
                  📦 প্রোডাক্ট তথ্য
                </h3>
                <div className="space-y-3">
                  {[
                    { label: "ক্যাটাগরি", value: product.category },
                    { label: "মূল্য", value: product.price },
                    { label: "ডেলিভারি", value: "ইনস্ট্যান্ট (অটোমেটিক)" },
                    { label: "পেমেন্ট", value: "bKash / Nagad / Card" },
                    { label: "সাপোর্ট", value: "২৪/৭ WhatsApp সাপোর্ট" },
                  ].map((item) => (
                    <div key={item.label} className="flex items-center justify-between py-2.5 border-b border-border/50 last:border-0">
                      <span className="text-sm text-muted-foreground">{item.label}</span>
                      <span className="text-sm font-semibold text-foreground">{item.value}</span>
                    </div>
                  ))}
                </div>
              </div>
            )}
          </motion.div>
        </div>
      </section>

      {/* Why Choose Us */}
      <section className="py-6 sm:py-10 bg-muted/20">
        <div className="container mx-auto px-3 sm:px-4">
          <h3 className="font-heading text-lg sm:text-xl font-bold text-foreground mb-5 text-center">
            🤝 কেন আমাদের থেকে কিনবেন?
          </h3>
          <div className="grid grid-cols-2 sm:grid-cols-4 gap-3">
            {[
              { icon: Shield, title: "১০০% অরিজিনাল", desc: "সব প্রোডাক্ট জেনুইন", color: "bg-emerald-500/10 text-emerald-600" },
              { icon: Zap, title: "ইনস্ট্যান্ট ডেলিভারি", desc: "পেমেন্টের পর সাথে সাথে", color: "bg-amber-500/10 text-amber-600" },
              { icon: MessageCircle, title: "২৪/৭ সাপোর্ট", desc: "যেকোনো সমস্যায় পাশে আছি", color: "bg-blue-500/10 text-blue-600" },
              { icon: ThumbsUp, title: "সন্তুষ্টি গ্যারান্টি", desc: "মানি ব্যাক গ্যারান্টি", color: "bg-purple-500/10 text-purple-600" },
            ].map(({ icon: Icon, title, desc, color }) => (
              <div key={title} className="flex flex-col items-center text-center p-4 bg-card rounded-2xl border border-border">
                <div className={`h-10 w-10 rounded-xl ${color} flex items-center justify-center mb-2.5`}>
                  <Icon className="h-5 w-5" />
                </div>
                <h4 className="text-xs sm:text-sm font-bold text-foreground mb-0.5">{title}</h4>
                <p className="text-[10px] sm:text-xs text-muted-foreground leading-tight">{desc}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* FAQ Section */}
      {faqs.length > 0 && (
        <section className="py-6 sm:py-10 bg-muted/20">
          <div className="container mx-auto px-3 sm:px-4">
            <h3 className="font-heading text-lg sm:text-xl font-bold text-foreground mb-5 flex items-center gap-2">
              ❓ সাধারণ জিজ্ঞাসা (FAQ)
            </h3>
            <div className="space-y-2.5">
              {faqs.map((faq) => (
                <div key={faq.id} className="bg-card rounded-xl border border-border overflow-hidden">
                  <button
                    onClick={() => setOpenFaqId(openFaqId === faq.id ? null : faq.id)}
                    className="w-full flex items-center justify-between px-4 py-3.5 text-left"
                  >
                    <span className="text-sm font-semibold text-foreground pr-2">{faq.question}</span>
                    <motion.div
                      animate={{ rotate: openFaqId === faq.id ? 180 : 0 }}
                      transition={{ duration: 0.2 }}
                    >
                      <ChevronDownIcon className="h-4 w-4 text-muted-foreground shrink-0" />
                    </motion.div>
                  </button>
                  <motion.div
                    initial={false}
                    animate={{
                      height: openFaqId === faq.id ? "auto" : 0,
                      opacity: openFaqId === faq.id ? 1 : 0,
                    }}
                    transition={{ duration: 0.25, ease: "easeInOut" }}
                    className="overflow-hidden"
                  >
                    <div className="px-4 pb-4 text-sm text-muted-foreground leading-relaxed border-t border-border/50 pt-3">
                      {faq.answer}
                    </div>
                  </motion.div>
                </div>
              ))}
            </div>
          </div>
        </section>
      )}


      <section className="py-6 sm:py-10">
        <div className="container mx-auto px-3 sm:px-4">
          <h3 className="font-heading text-lg sm:text-xl font-bold text-foreground mb-5 flex items-center gap-2">
            ⭐ কাস্টমার রিভিউ
            {reviews.length > 0 && (
              <span className="text-sm font-normal text-muted-foreground">({reviews.length}টি রিভিউ)</span>
            )}
          </h3>

          {/* Average Rating */}
          {reviews.length > 0 && (
            <div className="bg-card rounded-2xl border border-border p-5 mb-5 flex items-center gap-5">
              <div className="text-center">
                <p className="text-4xl font-bold text-foreground">
                  {(reviews.reduce((a: number, r: any) => a + r.rating, 0) / reviews.length).toFixed(1)}
                </p>
                <div className="flex items-center gap-0.5 mt-1 justify-center">
                  {[1, 2, 3, 4, 5].map(s => {
                    const avg = reviews.reduce((a: number, r: any) => a + r.rating, 0) / reviews.length;
                    return <Star key={s} className={`h-4 w-4 ${s <= Math.round(avg) ? "fill-yellow-400 text-yellow-400" : "text-muted-foreground/30"}`} />;
                  })}
                </div>
                <p className="text-xs text-muted-foreground mt-1">{reviews.length}টি রিভিউ</p>
              </div>
              <div className="flex-1 space-y-1">
                {[5, 4, 3, 2, 1].map(star => {
                  const count = reviews.filter((r: any) => r.rating === star).length;
                  const pct = reviews.length ? (count / reviews.length) * 100 : 0;
                  return (
                    <div key={star} className="flex items-center gap-2">
                      <span className="text-xs text-muted-foreground w-3">{star}</span>
                      <Star className="h-3 w-3 fill-yellow-400 text-yellow-400" />
                      <div className="flex-1 h-2 bg-muted rounded-full overflow-hidden">
                        <div className="h-full bg-yellow-400 rounded-full transition-all" style={{ width: `${pct}%` }} />
                      </div>
                      <span className="text-xs text-muted-foreground w-6 text-right">{count}</span>
                    </div>
                  );
                })}
              </div>
            </div>
          )}

          {/* Review List */}
          {reviews.length > 0 ? (
            <div className="space-y-3 mb-6">
              {reviews.map((review: any) => (
                <div key={review.id} className="bg-card rounded-xl border border-border p-4">
                  <div className="flex items-center gap-2 mb-2">
                    <div className="h-8 w-8 rounded-full bg-primary/10 flex items-center justify-center">
                      <span className="text-xs font-bold text-primary">{review.user_name?.[0]?.toUpperCase()}</span>
                    </div>
                    <div>
                      <p className="text-sm font-semibold text-foreground">{review.user_name}</p>
                      <p className="text-[10px] text-muted-foreground">{new Date(review.created_at).toLocaleDateString("bn-BD")}</p>
                    </div>
                    <div className="flex items-center gap-0.5 ml-auto">
                      {[1, 2, 3, 4, 5].map(s => (
                        <Star key={s} className={`h-3.5 w-3.5 ${s <= review.rating ? "fill-yellow-400 text-yellow-400" : "text-muted-foreground/30"}`} />
                      ))}
                    </div>
                  </div>
                  {review.comment && <p className="text-sm text-foreground/80 leading-relaxed">{review.comment}</p>}
                </div>
              ))}
            </div>
          ) : (
            <div className="bg-muted/30 rounded-xl border border-border p-6 text-center mb-6">
              <p className="text-sm text-muted-foreground">এখনো কোনো রিভিউ নেই। প্রথম রিভিউ দিন! 🎉</p>
            </div>
          )}

          {/* Write Review Form */}
          <div className="bg-card rounded-2xl border border-border p-5">
            <h4 className="font-heading text-base font-bold text-foreground mb-3">✍️ আপনার রিভিউ দিন</h4>
            <div className="mb-3">
              <label className="block text-sm font-semibold text-foreground mb-1.5">রেটিং</label>
              <div className="flex items-center gap-1">
                {[1, 2, 3, 4, 5].map(s => (
                  <button key={s} onClick={() => setReviewRating(s)} className="transition-transform hover:scale-110">
                    <Star className={`h-7 w-7 ${s <= reviewRating ? "fill-yellow-400 text-yellow-400" : "text-muted-foreground/30"}`} />
                  </button>
                ))}
                <span className="text-sm text-muted-foreground ml-2">{reviewRating}/5</span>
              </div>
            </div>
            <div className="mb-3">
              <label className="block text-sm font-semibold text-foreground mb-1.5">মতামত</label>
              <Textarea
                value={reviewComment}
                onChange={(e) => setReviewComment(e.target.value)}
                placeholder="আপনার অভিজ্ঞতা শেয়ার করুন..."
                rows={3}
                className="rounded-xl border-2 focus:border-primary resize-none"
              />
            </div>
            <Button
              onClick={handleSubmitReview}
              disabled={submittingReview}
              className="gap-2 rounded-xl"
            >
              {submittingReview ? <RefreshCw className="h-4 w-4 animate-spin" /> : <Send className="h-4 w-4" />}
              রিভিউ সাবমিট করুন
            </Button>
            <p className="text-[10px] text-muted-foreground mt-2">* রিভিউ অ্যাডমিন অ্যাপ্রুভ করলে প্রকাশিত হবে</p>
          </div>
        </div>
      </section>


      {relatedProducts.length > 0 && (
        <div className="border-t border-border">
          <ProductSection title="সম্পর্কিত প্রোডাক্ট" id="related" products={relatedProducts} />
        </div>
      )}

      <ShopFooter />
      <BottomBar />
    </div>
  );
};

export default ProductDetail;
