import { useState, useEffect, useRef } from "react";
import { useNavigate } from "react-router-dom";
import { supabase } from "@/integrations/supabase/client";
import { useAuth } from "@/contexts/AuthContext";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { toast } from "@/hooks/use-toast";
import { Avatar, AvatarImage, AvatarFallback } from "@/components/ui/avatar";
import { User, Phone, Mail, Package, LogOut, Camera, Key, Copy, CheckCircle2, ChevronDown, Shield, Store, CalendarDays, CreditCard, Hash, Sparkles, ShoppingBag, Award, Lock, Eye, EyeOff } from "lucide-react";
import { Tabs, TabsList, TabsTrigger, TabsContent } from "@/components/ui/tabs";
import { motion, AnimatePresence } from "framer-motion";
import Header from "@/components/Header";
import TopBar from "@/components/TopBar";

interface Order {
  id: string;
  product_name: string;
  amount: string;
  status: string;
  payment_status: string | null;
  payment_method: string | null;
  customer_name: string;
  customer_phone: string | null;
  transaction_id: string | null;
  created_at: string;
}

interface DeliveredKey {
  id: string;
  key_value: string;
  product_name: string;
  sold_at: string | null;
  order_id: string;
}

const Profile = () => {
  const { user, signOut, loading: authLoading, role, isReseller, isAdmin } = useAuth();
  const navigate = useNavigate();
  const [name, setName] = useState("");
  const [phone, setPhone] = useState("");
  const [email, setEmail] = useState("");
  const [avatarUrl, setAvatarUrl] = useState<string | null>(null);
  const [saving, setSaving] = useState(false);
  const [uploadingAvatar, setUploadingAvatar] = useState(false);
  const [orders, setOrders] = useState<Order[]>([]);
  const [deliveredKeys, setDeliveredKeys] = useState<DeliveredKey[]>([]);
  const [loadingProfile, setLoadingProfile] = useState(true);
  const [copiedKey, setCopiedKey] = useState<string | null>(null);
  const [expandedOrder, setExpandedOrder] = useState<string | null>(null);
  const [oldPassword, setOldPassword] = useState("");
  const [newPassword, setNewPassword] = useState("");
  const [confirmPassword, setConfirmPassword] = useState("");
  const [showPassword, setShowPassword] = useState(false);
  const [changingPassword, setChangingPassword] = useState(false);
  const fileInputRef = useRef<HTMLInputElement>(null);

  useEffect(() => {
    if (!authLoading && !user) {
      navigate("/auth");
    }
  }, [user, authLoading, navigate]);

  useEffect(() => {
    if (!authLoading && !user) {
      navigate("/auth");
    }
  }, [user, authLoading, navigate]);

  useEffect(() => {
    if (!user) return;
    setEmail(user.email || "");

    const fetchData = async () => {
      const [profileRes, ordersRes] = await Promise.all([
        supabase.from("profiles").select("name, phone, avatar_url").eq("user_id", user.id).single(),
        supabase.from("orders").select("*").eq("user_id", user.id).order("created_at", { ascending: false }),
      ]);

      if (profileRes.data) {
        setName(profileRes.data.name || "");
        setPhone(profileRes.data.phone || "");
        setAvatarUrl(profileRes.data.avatar_url || null);
      }
      if (ordersRes.data) {
        setOrders(ordersRes.data);
        const orderIds = ordersRes.data.map((o: any) => o.id);
        if (orderIds.length > 0) {
          const { data: keysData } = await supabase
            .from("product_keys")
            .select("id, key_value, sold_at, order_id")
            .in("order_id", orderIds);
          if (keysData) {
            const keysWithProducts = keysData.map((k: any) => {
              const order = ordersRes.data.find((o: any) => o.id === k.order_id);
              return {
                id: k.id,
                key_value: k.key_value,
                product_name: order?.product_name || "Unknown",
                sold_at: k.sold_at,
                order_id: k.order_id,
              };
            });
            setDeliveredKeys(keysWithProducts);
          }
        }
      }
      setLoadingProfile(false);
    };
    fetchData();
  }, [user]);

  const handleAvatarUpload = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file || !user) return;
    if (!file.type.startsWith("image/")) {
      toast({ title: "শুধুমাত্র ছবি আপলোড করুন", variant: "destructive" });
      return;
    }
    if (file.size > 2 * 1024 * 1024) {
      toast({ title: "ছবির সাইজ ২MB এর কম হতে হবে", variant: "destructive" });
      return;
    }
    setUploadingAvatar(true);
    const ext = file.name.split(".").pop();
    const filePath = `${user.id}/avatar.${ext}`;
    const { error: uploadError } = await supabase.storage.from("avatars").upload(filePath, file, { upsert: true });
    if (uploadError) {
      toast({ title: "আপলোড ব্যর্থ", description: uploadError.message, variant: "destructive" });
      setUploadingAvatar(false);
      return;
    }
    const { data: { publicUrl } } = supabase.storage.from("avatars").getPublicUrl(filePath);
    const url = `${publicUrl}?t=${Date.now()}`;
    await supabase.from("profiles").update({ avatar_url: url }).eq("user_id", user.id);
    setAvatarUrl(url);
    toast({ title: "প্রোফাইল ছবি আপডেট হয়েছে!" });
    setUploadingAvatar(false);
  };

  const handleSave = async () => {
    if (!user) return;
    setSaving(true);
    const { error: profileError } = await supabase.from("profiles").update({ name, phone }).eq("user_id", user.id);
    let emailError = null;
    if (email !== user.email) {
      const { error } = await supabase.auth.updateUser({ email });
      emailError = error;
    }
    if (profileError || emailError) {
      toast({ title: "Error", description: profileError?.message || emailError?.message, variant: "destructive" });
    } else {
      toast({
        title: "প্রোফাইল আপডেট হয়েছে!",
        description: email !== user.email ? "ইমেইল পরিবর্তনের জন্য নতুন ইমেইলে ভেরিফিকেশন লিংক পাঠানো হয়েছে।" : undefined,
      });
    }
    setSaving(false);
  };

  const handlePasswordChange = async () => {
    if (!oldPassword || !newPassword || !confirmPassword) {
      toast({ title: "সব ফিল্ড পূরণ করুন", variant: "destructive" });
      return;
    }
    if (newPassword.length < 6) {
      toast({ title: "পাসওয়ার্ড কমপক্ষে ৬ অক্ষরের হতে হবে", variant: "destructive" });
      return;
    }
    if (newPassword !== confirmPassword) {
      toast({ title: "পাসওয়ার্ড মিলছে না", variant: "destructive" });
      return;
    }
    setChangingPassword(true);
    // Verify old password first
    const { error: signInError } = await supabase.auth.signInWithPassword({
      email: user?.email || "",
      password: oldPassword,
    });
    if (signInError) {
      toast({ title: "পুরাতন পাসওয়ার্ড ভুল হয়েছে", variant: "destructive" });
      setChangingPassword(false);
      return;
    }
    const { error } = await supabase.auth.updateUser({ password: newPassword });
    if (error) {
      toast({ title: "পাসওয়ার্ড পরিবর্তন ব্যর্থ", description: error.message, variant: "destructive" });
    } else {
      toast({ title: "পাসওয়ার্ড সফলভাবে পরিবর্তন হয়েছে!" });
      setOldPassword("");
      setNewPassword("");
      setConfirmPassword("");
    }
    setChangingPassword(false);
  };

  const statusColor = (status: string) => {
    switch (status) {
      case "completed": return "bg-primary/15 text-primary border border-primary/20";
      case "pending": return "bg-accent/15 text-accent-foreground border border-accent/20";
      case "cancelled": return "bg-destructive/10 text-destructive border border-destructive/20";
      default: return "bg-muted text-muted-foreground";
    }
  };

  const roleBadge = () => {
    if (isAdmin) return { icon: <Shield className="h-3.5 w-3.5" />, label: "Admin", cls: "bg-destructive/10 text-destructive border-destructive/20" };
    if (isReseller) return { icon: <Store className="h-3.5 w-3.5" />, label: "Reseller", cls: "bg-primary/10 text-primary border-primary/20" };
    return { icon: <User className="h-3.5 w-3.5" />, label: "User", cls: "bg-muted text-muted-foreground border-border" };
  };

  if (authLoading || loadingProfile) {
    return (
      <div className="min-h-screen flex flex-col items-center justify-center bg-background gap-3">
        <motion.div
          animate={{ rotate: 360 }}
          transition={{ duration: 1, repeat: Infinity, ease: "linear" }}
          className="h-10 w-10 border-4 border-primary border-t-transparent rounded-full"
        />
        <p className="text-sm text-muted-foreground">লোড হচ্ছে...</p>
      </div>
    );
  }

  const badge = roleBadge();
  const stagger = { hidden: {}, show: { transition: { staggerChildren: 0.08 } } };
  const fadeUp = { hidden: { opacity: 0, y: 15 }, show: { opacity: 1, y: 0, transition: { duration: 0.4 } } };
  

  return (
    <div className="min-h-screen bg-gradient-to-b from-primary/10 via-primary/5 to-background">
      <TopBar />
      <Header />
      <div className="pb-24 md:pb-4">
        {/* ===== HERO SECTION ===== */}
        <div className="relative">
          <div className="absolute inset-0 bg-[radial-gradient(circle_at_30%_40%,hsl(var(--primary)/0.1),transparent_60%)]" />

          <div className="container mx-auto px-4 pt-3 sm:pt-6 pb-14 sm:pb-20 max-w-lg sm:max-w-2xl relative z-10">
            <motion.div
              variants={stagger}
              initial="hidden"
              animate="show"
              className="flex flex-col items-center"
            >
              {/* Avatar */}
              <motion.div variants={fadeUp} className="relative group cursor-pointer mb-3 sm:mb-5" onClick={() => fileInputRef.current?.click()}>
                <div className="absolute -inset-1.5 rounded-full bg-gradient-to-tr from-primary/40 via-primary/20 to-primary/40" style={{ animationDuration: '8s' }} />
                <div className="absolute -inset-1.5 rounded-full bg-background" style={{ margin: "2px" }} />
                <Avatar className="h-20 w-20 sm:h-28 sm:w-28 border-2 border-background shadow-xl relative z-10">
                  <AvatarImage src={avatarUrl || undefined} alt="Profile" />
                  <AvatarFallback className="bg-primary/10 text-primary text-2xl sm:text-3xl font-bold">
                    {name ? name.charAt(0).toUpperCase() : <User className="h-8 w-8" />}
                  </AvatarFallback>
                </Avatar>
                <div className="absolute inset-0 flex items-center justify-center bg-foreground/40 rounded-full opacity-0 group-hover:opacity-100 transition-all z-20">
                  <Camera className="h-5 w-5 text-background" />
                </div>
                {uploadingAvatar && (
                  <div className="absolute inset-0 flex items-center justify-center bg-foreground/50 rounded-full z-20">
                    <div className="animate-spin h-5 w-5 border-2 border-background border-t-transparent rounded-full" />
                  </div>
                )}
                <div className="absolute -bottom-0.5 -right-0.5 bg-primary text-primary-foreground rounded-full p-1.5 shadow-md border-2 border-background z-20">
                  <Camera className="h-3 w-3" />
                </div>
              </motion.div>
              <input ref={fileInputRef} type="file" accept="image/*" className="hidden" onChange={handleAvatarUpload} />

              <motion.h1 variants={fadeUp} className="text-xl sm:text-2xl font-bold text-foreground tracking-tight text-center">
                {name || "Your Name"}
              </motion.h1>
              <motion.p variants={fadeUp} className="text-xs sm:text-sm text-muted-foreground mt-0.5">{user?.email}</motion.p>

              <motion.div variants={fadeUp} className="flex items-center gap-1.5 mt-3">
                <span className={`inline-flex items-center gap-1 px-2.5 py-1 rounded-full text-[10px] font-semibold border ${badge.cls}`}>
                  {badge.icon} {badge.label}
                </span>
                <span className="inline-flex items-center gap-1 px-2.5 py-1 rounded-full text-[10px] font-semibold bg-primary/10 text-primary border border-primary/20">
                  <span className="relative flex h-1.5 w-1.5">
                    <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-primary opacity-50" />
                    <span className="relative inline-flex rounded-full h-1.5 w-1.5 bg-primary" />
                  </span>
                  Active
                </span>
              </motion.div>
            </motion.div>
          </div>
        </div>

        {/* ===== CONTENT ===== */}
        <div className="container mx-auto px-3 sm:px-4 max-w-lg sm:max-w-2xl -mt-6 sm:-mt-10 relative z-20">
          <Tabs defaultValue="profile" className="w-full">
            <TabsList className="w-full grid grid-cols-3 mb-4 bg-card/90 backdrop-blur-md border border-border shadow-md h-11 sm:h-13 rounded-xl sm:rounded-2xl p-1 sm:p-1.5">
              <TabsTrigger value="profile" className="gap-1 text-[11px] sm:text-sm rounded-lg sm:rounded-xl data-[state=active]:shadow-sm data-[state=active]:bg-primary data-[state=active]:text-primary-foreground transition-all">
                <User className="h-3.5 w-3.5" /> প্রোফাইল
              </TabsTrigger>
              <TabsTrigger value="orders" className="gap-1 text-[11px] sm:text-sm rounded-lg sm:rounded-xl data-[state=active]:shadow-sm data-[state=active]:bg-primary data-[state=active]:text-primary-foreground transition-all">
                <Package className="h-3.5 w-3.5" /> অর্ডার
              </TabsTrigger>
              <TabsTrigger value="keys" className="gap-1 text-[11px] sm:text-sm rounded-lg sm:rounded-xl data-[state=active]:shadow-sm data-[state=active]:bg-primary data-[state=active]:text-primary-foreground transition-all">
                <Key className="h-3.5 w-3.5" /> কী সমূহ
              </TabsTrigger>
            </TabsList>

            {/* ===== PROFILE TAB ===== */}
            <TabsContent value="profile">
              <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} transition={{ duration: 0.3 }} className="space-y-3">
                {/* Quick Stats */}
                <div className="grid grid-cols-3 gap-2">
                  {[
                    { icon: <ShoppingBag className="h-4 w-4" />, value: orders.length, label: "অর্ডার" },
                    { icon: <CheckCircle2 className="h-4 w-4" />, value: orders.filter(o => o.status === "completed").length, label: "সম্পন্ন" },
                    { icon: <Award className="h-4 w-4" />, value: deliveredKeys.length, label: "লাইসেন্স" },
                  ].map((stat, i) => (
                    <div
                      key={i}
                      className="bg-card border border-border rounded-xl p-3 text-center hover:border-primary/30 transition-colors"
                    >
                      <div className="text-primary/60 mb-1 flex justify-center">{stat.icon}</div>
                      <p className="text-2xl font-bold text-foreground">{stat.value}</p>
                      <p className="text-[9px] text-muted-foreground font-medium mt-0.5">{stat.label}</p>
                    </div>
                  ))}
                </div>

                {/* Form Card */}
                <div className="bg-card border border-border rounded-xl shadow-sm overflow-hidden">
                  <div className="px-4 py-3 border-b border-border bg-gradient-to-r from-primary/5 to-transparent">
                    <h2 className="text-sm font-bold text-foreground flex items-center gap-2">
                      <Sparkles className="h-4 w-4 text-primary" />
                      প্রোফাইল সেটিংস
                    </h2>
                    <p className="text-[10px] text-muted-foreground mt-0.5 ml-6">আপনার ব্যক্তিগত তথ্য আপডেট করুন</p>
                  </div>

                  <div className="p-4 space-y-3">
                    {[
                      { icon: <User className="h-4 w-4" />, label: "নাম", placeholder: "আপনার নাম", value: name, onChange: (v: string) => setName(v), type: "text" },
                      { icon: <Mail className="h-4 w-4" />, label: "ইমেইল", placeholder: "ইমেইল এড্রেস", value: email, onChange: (v: string) => setEmail(v), type: "email" },
                      { icon: <Phone className="h-4 w-4" />, label: "ফোন", placeholder: "ফোন নম্বর", value: phone, onChange: (v: string) => setPhone(v), type: "tel" },
                    ].map((field, i) => (
                      <div key={i}>
                        <label className="text-[10px] font-semibold text-muted-foreground mb-1.5 block uppercase tracking-wider">{field.label}</label>
                        <div className="relative group">
                          <div className="absolute left-0 top-0 bottom-0 w-10 flex items-center justify-center text-muted-foreground group-focus-within:text-primary transition-colors">
                            {field.icon}
                          </div>
                          <Input
                            placeholder={field.placeholder}
                            type={field.type}
                            value={field.value}
                            onChange={(e) => field.onChange(e.target.value)}
                            className="pl-10 h-11 rounded-lg border-border bg-muted/30 focus:bg-background transition-colors text-sm"
                          />
                        </div>
                      </div>
                    ))}

                    <div className="flex gap-2 pt-2">
                      <Button
                        onClick={handleSave}
                        disabled={saving}
                        className="flex-1 h-11 rounded-lg font-semibold text-sm"
                      >
                        {saving ? (
                          <span className="flex items-center gap-2">
                            <div className="animate-spin h-4 w-4 border-2 border-primary-foreground border-t-transparent rounded-full" />
                            সেভ হচ্ছে...
                          </span>
                        ) : (
                          <span className="flex items-center gap-2">
                            <CheckCircle2 className="h-4 w-4" />
                            আপডেট করুন
                          </span>
                        )}
                      </Button>
                      <Button
                        variant="outline"
                        onClick={async () => { await signOut(); navigate("/"); }}
                        className="text-destructive border-destructive/30 hover:bg-destructive hover:text-destructive-foreground h-11 rounded-lg"
                      >
                        <LogOut className="h-4 w-4" />
                      </Button>
                    </div>
                  </div>
                </div>

                {/* Password Change Card */}
                <div className="bg-card border border-border rounded-xl shadow-sm overflow-hidden">
                  <div className="px-4 py-3 border-b border-border bg-gradient-to-r from-primary/5 to-transparent">
                    <h2 className="text-sm font-bold text-foreground flex items-center gap-2">
                      <Lock className="h-4 w-4 text-primary" />
                      পাসওয়ার্ড পরিবর্তন
                    </h2>
                    <p className="text-[10px] text-muted-foreground mt-0.5 ml-6">আপনার পাসওয়ার্ড আপডেট করুন</p>
                  </div>
                  <div className="p-4 space-y-3">
                    {[
                      { label: "বর্তমান পাসওয়ার্ড", placeholder: "বর্তমান পাসওয়ার্ড দিন", value: oldPassword, onChange: setOldPassword },
                      { label: "নতুন পাসওয়ার্ড", placeholder: "নতুন পাসওয়ার্ড (মিনিমাম ৬ অক্ষর)", value: newPassword, onChange: setNewPassword, showToggle: true },
                      { label: "পাসওয়ার্ড নিশ্চিত করুন", placeholder: "আবার পাসওয়ার্ড দিন", value: confirmPassword, onChange: setConfirmPassword },
                    ].map((field, i) => (
                      <div key={i}>
                        <label className="text-[10px] font-semibold text-muted-foreground mb-1.5 block uppercase tracking-wider">{field.label}</label>
                        <div className="relative group">
                          <div className="absolute left-0 top-0 bottom-0 w-10 flex items-center justify-center text-muted-foreground group-focus-within:text-primary transition-colors">
                            <Lock className="h-4 w-4" />
                          </div>
                          <Input
                            placeholder={field.placeholder}
                            type={showPassword ? "text" : "password"}
                            value={field.value}
                            onChange={(e) => field.onChange(e.target.value)}
                            className="pl-10 pr-10 h-11 rounded-lg border-border bg-muted/30 focus:bg-background transition-colors text-sm"
                          />
                          {field.showToggle && (
                            <button
                              type="button"
                              onClick={() => setShowPassword(!showPassword)}
                              className="absolute right-3 top-1/2 -translate-y-1/2 text-muted-foreground hover:text-foreground transition-colors"
                            >
                              {showPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                            </button>
                          )}
                        </div>
                        {i === 2 && confirmPassword && newPassword !== confirmPassword && (
                          <p className="text-[10px] text-destructive mt-1 font-medium">পাসওয়ার্ড মিলছে না</p>
                        )}
                      </div>
                    ))}
                    <Button
                      onClick={handlePasswordChange}
                      disabled={changingPassword || !oldPassword || !newPassword || !confirmPassword}
                      className="w-full h-11 rounded-lg font-semibold text-sm"
                    >
                      {changingPassword ? (
                        <span className="flex items-center gap-2">
                          <div className="animate-spin h-4 w-4 border-2 border-primary-foreground border-t-transparent rounded-full" />
                          পরিবর্তন হচ্ছে...
                        </span>
                      ) : (
                        <span className="flex items-center gap-2">
                          <Lock className="h-4 w-4" />
                          পাসওয়ার্ড পরিবর্তন করুন
                        </span>
                      )}
                    </Button>
                  </div>
                </div>
              </motion.div>
            </TabsContent>

            {/* ===== ORDERS TAB ===== */}
            <TabsContent value="orders">
              <motion.div initial={{ opacity: 0, y: 15 }} animate={{ opacity: 1, y: 0 }} transition={{ duration: 0.4 }}>
                <div className="bg-card border border-border rounded-2xl shadow-sm overflow-hidden">
                  <div className="relative px-5 py-4 border-b border-border overflow-hidden">
                    <div className="absolute inset-0 bg-gradient-to-r from-primary/8 via-primary/3 to-transparent" />
                    <h2 className="font-heading text-lg font-bold text-foreground flex items-center gap-2 relative z-10">
                      <div className="w-8 h-8 rounded-xl bg-primary/10 flex items-center justify-center">
                        <Package className="h-4 w-4 text-primary" />
                      </div>
                      Order History
                    </h2>
                    <p className="text-xs text-muted-foreground mt-1 relative z-10 ml-10">
                      মোট {orders.length}টি অর্ডার
                    </p>
                  </div>

                  {orders.length === 0 ? (
                    <div className="text-center py-20 px-4">
                      <motion.div
                        initial={{ scale: 0.8, opacity: 0 }}
                        animate={{ scale: 1, opacity: 1 }}
                        transition={{ type: "spring", stiffness: 200 }}
                      >
                        <div className="w-20 h-20 mx-auto mb-5 rounded-2xl bg-gradient-to-br from-muted to-muted/50 flex items-center justify-center">
                          <Package className="h-10 w-10 text-muted-foreground/30" />
                        </div>
                         <p className="font-heading font-bold text-foreground text-lg">No Orders Yet</p>
                         <p className="text-sm text-muted-foreground mt-1.5">আপনার অর্ডার এখানে দেখা যাবে</p>
                         <Button onClick={() => navigate("/shop")} className="mt-5 rounded-xl shadow-md" size="sm">
                           <ShoppingBag className="h-4 w-4 mr-1.5" /> Go to Shop
                         </Button>
                      </motion.div>
                    </div>
                  ) : (
                    <div className="divide-y divide-border">
                      {orders.map((order, idx) => {
                        const orderKeys = deliveredKeys.filter(k => k.order_id === order.id);
                        const isExpanded = expandedOrder === order.id;
                        return (
                          <motion.div
                            key={order.id}
                            initial={{ opacity: 0 }}
                            animate={{ opacity: 1 }}
                            transition={{ delay: idx * 0.04 }}
                          >
                            <button
                              className="w-full flex items-center justify-between p-4 sm:p-5 hover:bg-muted/20 transition-colors text-left"
                              onClick={() => setExpandedOrder(isExpanded ? null : order.id)}
                            >
                              <div className="flex items-start gap-3 flex-1 min-w-0">
                                <div className={`w-10 h-10 rounded-xl flex items-center justify-center flex-shrink-0 ${
                                  order.status === "completed" ? "bg-primary/10" : "bg-muted"
                                }`}>
                                  <Package className={`h-5 w-5 ${order.status === "completed" ? "text-primary" : "text-muted-foreground"}`} />
                                </div>
                                <div className="flex-1 min-w-0">
                                  <p className="text-sm font-semibold text-foreground truncate">{order.product_name}</p>
                                  <div className="flex items-center gap-2 mt-1 flex-wrap">
                                    <span className="font-mono text-[10px] text-muted-foreground">#{order.id.slice(0, 8)}</span>
                                    <span className={`text-[10px] font-semibold px-2 py-0.5 rounded-full ${statusColor(order.status)}`}>
                                      {order.status === "completed" ? "✓ সম্পন্ন" : order.status === "pending" ? "⏳ পেন্ডিং" : order.status}
                                    </span>
                                  </div>
                                  <p className="text-[10px] text-muted-foreground mt-1 flex items-center gap-1">
                                    <CalendarDays className="h-3 w-3" />
                                    {new Date(order.created_at).toLocaleDateString("bn-BD", { year: "numeric", month: "short", day: "numeric" })}
                                  </p>
                                </div>
                              </div>
                              <div className="flex items-center gap-3 ml-3 flex-shrink-0">
                                <div className="text-right">
                                  <span className="font-bold text-foreground text-sm block">৳{order.amount}</span>
                                  {orderKeys.length > 0 && (
                                    <span className="text-[9px] text-primary font-semibold">🔑 কী আছে</span>
                                  )}
                                </div>
                                <motion.div animate={{ rotate: isExpanded ? 180 : 0 }} transition={{ duration: 0.2 }}>
                                  <ChevronDown className="h-4 w-4 text-muted-foreground" />
                                </motion.div>
                              </div>
                            </button>

                            <AnimatePresence>
                              {isExpanded && (
                                <motion.div
                                  initial={{ height: 0, opacity: 0 }}
                                  animate={{ height: "auto", opacity: 1 }}
                                  exit={{ height: 0, opacity: 0 }}
                                  transition={{ duration: 0.25, ease: "easeInOut" }}
                                  className="overflow-hidden"
                                >
                                  <div className="border-t border-border p-4 sm:p-5 space-y-3 bg-muted/10">
                                    <div className="grid grid-cols-2 gap-3">
                                      <div className="bg-card rounded-xl p-3.5 border border-border">
                                        <div className="flex items-center gap-1.5 mb-1">
                                          <CreditCard className="h-3 w-3 text-primary" />
                                          <p className="text-[10px] text-muted-foreground font-semibold uppercase tracking-wider">পেমেন্ট</p>
                                        </div>
                                        <p className="text-sm font-bold text-foreground">{order.payment_method || "N/A"}</p>
                                      </div>
                                      <div className="bg-card rounded-xl p-3.5 border border-border">
                                        <div className="flex items-center gap-1.5 mb-1">
                                          <Hash className="h-3 w-3 text-primary" />
                                          <p className="text-[10px] text-muted-foreground font-semibold uppercase tracking-wider">মোট</p>
                                        </div>
                                        <p className="text-sm font-bold text-foreground">৳{order.amount}</p>
                                      </div>
                                    </div>

                                    <div className="bg-card rounded-xl p-3.5 border border-border">
                                      <p className="text-[10px] text-muted-foreground font-semibold uppercase tracking-wider mb-2">কাস্টমার তথ্য</p>
                                      <p className="text-sm text-foreground font-medium">{order.customer_name}</p>
                                      {order.customer_phone && <p className="text-xs text-muted-foreground mt-0.5">{order.customer_phone}</p>}
                                      {order.transaction_id && <p className="text-xs text-muted-foreground font-mono mt-0.5">TXN: {order.transaction_id}</p>}
                                    </div>

                                    {orderKeys.length > 0 && (
                                      <div className="bg-primary/5 rounded-xl p-3.5 border border-primary/15">
                                        <p className="text-[10px] text-primary font-bold uppercase tracking-wider mb-2 flex items-center gap-1.5">
                                          <Key className="h-3 w-3" /> লাইসেন্স কী
                                        </p>
                                        <div className="space-y-2">
                                          {orderKeys.map((dk) => (
                                            <div key={dk.id} className="bg-card rounded-lg p-3 flex items-center gap-2 border border-border">
                                              <code className="text-xs font-mono flex-1 break-all text-foreground">{dk.key_value}</code>
                                              <button
                                                onClick={(e) => {
                                                  e.stopPropagation();
                                                  navigator.clipboard.writeText(dk.key_value);
                                                  setCopiedKey(dk.id);
                                                  setTimeout(() => setCopiedKey(null), 2000);
                                                }}
                                                className="flex-shrink-0 p-2 rounded-lg hover:bg-muted transition-colors"
                                              >
                                                {copiedKey === dk.id ? <CheckCircle2 className="h-4 w-4 text-primary" /> : <Copy className="h-4 w-4 text-muted-foreground" />}
                                              </button>
                                            </div>
                                          ))}
                                        </div>
                                      </div>
                                    )}

                                    <p className="text-[8px] text-muted-foreground/40 font-mono break-all pt-1">ID: {order.id}</p>
                                  </div>
                                </motion.div>
                              )}
                            </AnimatePresence>
                          </motion.div>
                        );
                      })}
                    </div>
                  )}
                </div>
              </motion.div>
            </TabsContent>

            {/* ===== KEYS TAB ===== */}
            <TabsContent value="keys">
              <motion.div initial={{ opacity: 0, y: 15 }} animate={{ opacity: 1, y: 0 }} transition={{ duration: 0.4 }}>
                <div className="bg-card border border-border rounded-2xl shadow-sm overflow-hidden">
                  <div className="relative px-5 py-4 border-b border-border overflow-hidden">
                    <div className="absolute inset-0 bg-gradient-to-r from-primary/8 via-primary/3 to-transparent" />
                    <h2 className="font-heading text-lg font-bold text-foreground flex items-center gap-2 relative z-10">
                      <div className="w-8 h-8 rounded-xl bg-primary/10 flex items-center justify-center">
                        <Key className="h-4 w-4 text-primary" />
                      </div>
                      আমার লাইসেন্স সমূহ
                    </h2>
                    <p className="text-xs text-muted-foreground mt-1 relative z-10 ml-10">
                      মোট {deliveredKeys.length}টি লাইসেন্স
                    </p>
                  </div>

                  {deliveredKeys.length === 0 ? (
                    <div className="text-center py-20 px-4">
                      <motion.div
                        initial={{ scale: 0.8, opacity: 0 }}
                        animate={{ scale: 1, opacity: 1 }}
                        transition={{ type: "spring", stiffness: 200 }}
                      >
                        <div className="w-20 h-20 mx-auto mb-5 rounded-2xl bg-gradient-to-br from-muted to-muted/50 flex items-center justify-center">
                          <Key className="h-10 w-10 text-muted-foreground/30" />
                        </div>
                        <p className="font-heading font-bold text-foreground text-lg">এখনো কোনো লাইসেন্স নেই</p>
                        <p className="text-sm text-muted-foreground mt-1.5">অর্ডার সম্পন্ন হলে আপনার লাইসেন্স এখানে দেখা যাবে</p>
                      </motion.div>
                    </div>
                  ) : (
                    <div className="p-4 space-y-3">
                      {deliveredKeys.map((dk, idx) => (
                        <motion.div
                          key={dk.id}
                          initial={{ opacity: 0, y: 8 }}
                          animate={{ opacity: 1, y: 0 }}
                          transition={{ delay: idx * 0.06 }}
                          className="group border border-border rounded-2xl overflow-hidden hover:border-primary/30 hover:shadow-md transition-all duration-300"
                        >
                          <div className="relative px-4 py-3 flex items-center justify-between overflow-hidden">
                            <div className="absolute inset-0 bg-gradient-to-r from-primary/5 to-transparent" />
                            <div className="flex items-center gap-2.5 min-w-0 relative z-10">
                              <div className="w-8 h-8 rounded-xl bg-primary/10 flex items-center justify-center flex-shrink-0 group-hover:bg-primary/20 transition-colors">
                                <Key className="h-3.5 w-3.5 text-primary" />
                              </div>
                              <span className="font-semibold text-foreground text-sm truncate">{dk.product_name}</span>
                            </div>
                            {dk.sold_at && (
                              <span className="text-[10px] text-muted-foreground whitespace-nowrap ml-2 flex items-center gap-1 relative z-10">
                                <CalendarDays className="h-3 w-3" />
                                {new Date(dk.sold_at).toLocaleDateString("bn-BD", { year: "numeric", month: "short", day: "numeric" })}
                              </span>
                            )}
                          </div>
                          <div className="px-4 pb-4">
                            <div className="bg-muted/30 border border-border/50 rounded-xl p-3.5 flex items-start gap-2 group-hover:bg-muted/40 transition-colors">
                              <code className="text-xs font-mono flex-1 break-all text-foreground leading-relaxed select-all">{dk.key_value}</code>
                              <motion.button
                                whileTap={{ scale: 0.9 }}
                                onClick={() => {
                                  navigator.clipboard.writeText(dk.key_value);
                                  setCopiedKey(dk.id);
                                  setTimeout(() => setCopiedKey(null), 2000);
                                }}
                                className={`flex-shrink-0 p-2.5 rounded-xl transition-all ${
                                  copiedKey === dk.id
                                    ? "bg-primary/15 text-primary"
                                    : "hover:bg-card text-muted-foreground hover:text-foreground"
                                }`}
                              >
                                {copiedKey === dk.id ? <CheckCircle2 className="h-4 w-4" /> : <Copy className="h-4 w-4" />}
                              </motion.button>
                            </div>
                            <AnimatePresence>
                              {copiedKey === dk.id && (
                                <motion.p
                                  initial={{ opacity: 0, y: -5 }}
                                  animate={{ opacity: 1, y: 0 }}
                                  exit={{ opacity: 0 }}
                                  className="text-[10px] text-primary mt-2 font-bold flex items-center gap-1"
                                >
                                  <CheckCircle2 className="h-3 w-3" /> কপি করা হয়েছে!
                                </motion.p>
                              )}
                            </AnimatePresence>
                          </div>
                        </motion.div>
                      ))}
                    </div>
                  )}
                </div>
              </motion.div>
            </TabsContent>
          </Tabs>
        </div>
      </div>
    </div>
  );
};

export default Profile;
