import { useState, useEffect, useMemo } from "react";
import { useNavigate } from "react-router-dom";
import { supabase } from "@/integrations/supabase/client";
import { useAuth } from "@/contexts/AuthContext";
import TopBar from "@/components/TopBar";
import Header from "@/components/Header";
import BottomBar from "@/components/BottomBar";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import {
  Table, TableBody, TableCell, TableHead, TableHeader, TableRow,
} from "@/components/ui/table";
import {
  BarChart3, ShoppingCart, TrendingUp, Package, DollarSign, Calendar, Loader2,
} from "lucide-react";

interface ResellerOrder {
  id: string;
  product_name: string;
  amount: string;
  status: string;
  payment_status: string | null;
  created_at: string;
}

interface ResellerProduct {
  id: string;
  name: string;
  price: string;
  reseller_price: string | null;
}

const parsePrice = (p: string): number => parseFloat(p.replace(/[^\d.]/g, "")) || 0;

const ResellerDashboard = () => {
  const { user, loading: authLoading, isReseller } = useAuth();
  const navigate = useNavigate();
  const [orders, setOrders] = useState<ResellerOrder[]>([]);
  const [products, setProducts] = useState<ResellerProduct[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    if (!authLoading && (!user || !isReseller)) navigate("/");
  }, [user, authLoading, isReseller, navigate]);

  useEffect(() => {
    if (!user) return;
    const fetchData = async () => {
      const [ordersRes, productsRes] = await Promise.all([
        supabase.from("orders").select("id, product_name, amount, status, payment_status, created_at")
          .eq("user_id", user.id).order("created_at", { ascending: false }),
        supabase.from("products").select("id, name, price, reseller_price"),
      ]);
      if (ordersRes.data) setOrders(ordersRes.data);
      if (productsRes.data) setProducts(productsRes.data);
      setLoading(false);
    };
    fetchData();
  }, [user]);

  const stats = useMemo(() => {
    const totalOrders = orders.length;
    const completedOrders = orders.filter(o => o.status === "completed");
    const totalSpent = completedOrders.reduce((s, o) => s + parsePrice(o.amount), 0);

    // Calculate potential profit: for each completed order, find the product and compare retail vs reseller price
    let totalProfit = 0;
    completedOrders.forEach(order => {
      const product = products.find(p => order.product_name.includes(p.name));
      if (product && product.reseller_price) {
        const retailPrice = parsePrice(product.price);
        const resellerPrice = parsePrice(product.reseller_price);
        totalProfit += retailPrice - resellerPrice;
      }
    });

    const thisMonth = orders.filter(o => {
      const d = new Date(o.created_at);
      const now = new Date();
      return d.getMonth() === now.getMonth() && d.getFullYear() === now.getFullYear();
    }).length;

    return { totalOrders, totalSpent, totalProfit, thisMonth };
  }, [orders, products]);

  if (authLoading || loading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-background">
        <Loader2 className="h-8 w-8 animate-spin text-primary" />
      </div>
    );
  }

  const statusBadge = (status: string) => {
    switch (status) {
      case "completed": return <Badge className="bg-primary/15 text-primary border-0 text-[10px]">সম্পন্ন</Badge>;
      case "pending": return <Badge className="bg-accent/50 text-accent-foreground border-0 text-[10px]">পেন্ডিং</Badge>;
      case "cancelled": return <Badge className="bg-destructive/15 text-destructive border-0 text-[10px]">বাতিল</Badge>;
      default: return <Badge variant="secondary" className="text-[10px]">{status}</Badge>;
    }
  };

  const resellerProducts = products.filter(p => p.reseller_price);

  return (
    <>
      <TopBar />
      <Header />
      <div className="min-h-screen bg-secondary/30 pb-20">
        <div className="container mx-auto px-3 sm:px-4 py-4 sm:py-8 max-w-5xl">
          {/* Header */}
          <div className="flex items-center gap-3 mb-6">
            <div className="h-10 w-10 rounded-xl bg-primary/10 flex items-center justify-center">
              <BarChart3 className="h-5 w-5 text-primary" />
            </div>
            <div>
              <h1 className="font-heading text-xl font-bold text-foreground">রিসেলার ড্যাশবোর্ড</h1>
              <p className="text-xs text-muted-foreground">আপনার সেলস ও প্রফিট ট্র্যাক করুন</p>
            </div>
          </div>

          {/* Stats Cards */}
          <div className="grid grid-cols-2 lg:grid-cols-4 gap-3 mb-6">
            <Card className="border-border/50">
              <CardContent className="p-4">
                <div className="flex items-center gap-2 mb-2">
                  <ShoppingCart className="h-4 w-4 text-primary" />
                  <span className="text-xs text-muted-foreground">মোট অর্ডার</span>
                </div>
                <p className="text-2xl font-bold text-foreground">{stats.totalOrders}</p>
              </CardContent>
            </Card>
            <Card className="border-border/50">
              <CardContent className="p-4">
                <div className="flex items-center gap-2 mb-2">
                  <DollarSign className="h-4 w-4 text-primary" />
                  <span className="text-xs text-muted-foreground">মোট খরচ</span>
                </div>
                <p className="text-2xl font-bold text-foreground">৳{stats.totalSpent}</p>
              </CardContent>
            </Card>
            <Card className="border-border/50">
              <CardContent className="p-4">
                <div className="flex items-center gap-2 mb-2">
                  <TrendingUp className="h-4 w-4 text-primary" />
                  <span className="text-xs text-muted-foreground">আনুমানিক প্রফিট</span>
                </div>
                <p className="text-2xl font-bold text-primary">৳{stats.totalProfit}</p>
              </CardContent>
            </Card>
            <Card className="border-border/50">
              <CardContent className="p-4">
                <div className="flex items-center gap-2 mb-2">
                  <Calendar className="h-4 w-4 text-primary" />
                  <span className="text-xs text-muted-foreground">এই মাসে</span>
                </div>
                <p className="text-2xl font-bold text-foreground">{stats.thisMonth}</p>
              </CardContent>
            </Card>
          </div>

          {/* Tabs */}
          <Tabs defaultValue="orders">
            <TabsList className="w-full justify-start mb-4">
              <TabsTrigger value="orders" className="gap-1.5 text-xs"><Package className="h-3.5 w-3.5" /> অর্ডার হিস্ট্রি</TabsTrigger>
              <TabsTrigger value="pricing" className="gap-1.5 text-xs"><DollarSign className="h-3.5 w-3.5" /> রিসেলার প্রাইসিং</TabsTrigger>
            </TabsList>

            <TabsContent value="orders">
              <Card>
                <CardHeader className="pb-3">
                  <CardTitle className="text-sm font-semibold">সকল অর্ডার</CardTitle>
                </CardHeader>
                <CardContent className="p-0">
                  {orders.length === 0 ? (
                    <div className="p-8 text-center text-muted-foreground text-sm">
                      <Package className="h-10 w-10 mx-auto mb-2 opacity-30" />
                      এখনো কোনো অর্ডার নেই
                    </div>
                  ) : (
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead className="text-xs">অর্ডার</TableHead>
                          <TableHead className="text-xs">প্রোডাক্ট</TableHead>
                          <TableHead className="text-xs">মূল্য</TableHead>
                          <TableHead className="text-xs">স্ট্যাটাস</TableHead>
                          <TableHead className="text-xs">তারিখ</TableHead>
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {orders.map((order) => {
                          const product = products.find(p => order.product_name.includes(p.name));
                          const profit = product?.reseller_price
                            ? parsePrice(product.price) - parsePrice(product.reseller_price)
                            : 0;
                          return (
                            <TableRow key={order.id}>
                              <TableCell className="text-xs font-mono">#{order.id.slice(0, 8)}</TableCell>
                              <TableCell className="text-xs max-w-[150px] truncate">{order.product_name}</TableCell>
                              <TableCell className="text-xs">
                                <span className="font-semibold">{order.amount}</span>
                                {profit > 0 && order.status === "completed" && (
                                  <span className="block text-[10px] text-primary">+৳{profit} প্রফিট</span>
                                )}
                              </TableCell>
                              <TableCell>{statusBadge(order.status)}</TableCell>
                              <TableCell className="text-xs text-muted-foreground">
                                {new Date(order.created_at).toLocaleDateString("bn-BD", { day: "numeric", month: "short" })}
                              </TableCell>
                            </TableRow>
                          );
                        })}
                      </TableBody>
                    </Table>
                  )}
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="pricing">
              <Card>
                <CardHeader className="pb-3">
                  <CardTitle className="text-sm font-semibold">🏪 রিসেলার প্রাইস তালিকা</CardTitle>
                </CardHeader>
                <CardContent className="p-0">
                  {resellerProducts.length === 0 ? (
                    <div className="p-8 text-center text-muted-foreground text-sm">
                      রিসেলার প্রাইসিং সেট করা হয়নি
                    </div>
                  ) : (
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead className="text-xs">প্রোডাক্ট</TableHead>
                          <TableHead className="text-xs">রিটেইল মূল্য</TableHead>
                          <TableHead className="text-xs">🏪 রিসেলার মূল্য</TableHead>
                          <TableHead className="text-xs">প্রফিট মার্জিন</TableHead>
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {resellerProducts.map((p) => {
                          const retail = parsePrice(p.price);
                          const reseller = parsePrice(p.reseller_price!);
                          const margin = retail - reseller;
                          const pct = retail > 0 ? Math.round((margin / retail) * 100) : 0;
                          return (
                            <TableRow key={p.id}>
                              <TableCell className="text-xs font-medium">{p.name}</TableCell>
                              <TableCell className="text-xs">{p.price}</TableCell>
                              <TableCell className="text-xs font-semibold text-primary">{p.reseller_price}</TableCell>
                              <TableCell className="text-xs">
                                <span className="font-semibold text-primary">৳{margin}</span>
                                <span className="text-muted-foreground ml-1">({pct}%)</span>
                              </TableCell>
                            </TableRow>
                          );
                        })}
                      </TableBody>
                    </Table>
                  )}
                </CardContent>
              </Card>
            </TabsContent>
          </Tabs>
        </div>
      </div>
      <BottomBar />
    </>
  );
};

export default ResellerDashboard;
