import { useState, useMemo, useEffect } from "react";
import TopBar from "@/components/TopBar";
import Header from "@/components/Header";
import ShopFooter from "@/components/ShopFooter";
import BottomBar from "@/components/BottomBar";
import ShopProductCard from "@/components/ShopProductCard";
import { useSiteSettings } from "@/hooks/useSiteSettings";
import { supabase } from "@/integrations/supabase/client";
import type { Product } from "@/components/productsData";
import { Search, SlidersHorizontal, X } from "lucide-react";
import { Input } from "@/components/ui/input";
import { motion, AnimatePresence } from "framer-motion";

const Shop = () => {
  useSiteSettings();
  const [activeCategory, setActiveCategory] = useState("All");
  const [searchQuery, setSearchQuery] = useState("");
  const [allProducts, setAllProducts] = useState<Product[]>([]);
  const [categories, setCategories] = useState<string[]>([]);

  useEffect(() => {
    const fetchData = async () => {
      const [catRes, prodRes] = await Promise.all([
        supabase.from("categories").select("name").eq("is_active", true).order("sort_order"),
        supabase.from("products").select("*"),
      ]);
      if (catRes.data) setCategories(catRes.data.map((c) => c.name));
      if (prodRes.data) {
        setAllProducts(
          prodRes.data.map((p: any) => ({
            id: p.id,
            name: p.name,
            slug: p.slug || p.id,
            image: p.image_url || "/placeholder.svg",
            price: p.price,
            originalPrice: p.original_price || undefined,
            resellerPrice: p.reseller_price || undefined,
            resellerPrices: p.reseller_prices || undefined,
            category: p.category,
            badges: (p.badges || []) as ("sale" | "limited")[],
            description: p.description || undefined,
            features: p.features || undefined,
            duration: p.duration_options || undefined,
          }))
        );
      }
    };
    fetchData();
  }, []);

  const filteredProducts = useMemo(() => {
    let products = allProducts;
    if (activeCategory !== "All") {
      products = products.filter((p) => p.category.split(",").map(s => s.trim()).includes(activeCategory));
    }
    if (searchQuery.trim()) {
      const q = searchQuery.toLowerCase();
      products = products.filter(
        (p) =>
          p.name.toLowerCase().includes(q) ||
          p.category.toLowerCase().includes(q) ||
          p.description?.toLowerCase().includes(q)
      );
    }
    return products;
  }, [activeCategory, searchQuery, allProducts]);

  return (
    <div className="min-h-screen bg-gradient-to-b from-primary/10 via-background to-background">
      <TopBar />
      <Header />

      {/* Hero Section */}
      <div className="border-b border-border">
        <div className="container mx-auto px-4 py-8 md:py-12">
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.5 }}
            className="text-center max-w-2xl mx-auto"
          >
            <h1 className="font-heading text-3xl md:text-4xl font-bold text-foreground mb-3">
              সকল প্রোডাক্ট
            </h1>
            <p className="text-muted-foreground text-sm md:text-base mb-6">
              প্রিমিয়াম ডিজিটাল প্রোডাক্ট সেরা দামে — সফটওয়্যার, VPN, AI টুলস, গিফট কার্ড
            </p>

            {/* Search */}
            <div className="relative max-w-md mx-auto">
              <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
              <Input
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                placeholder="প্রোডাক্ট খুঁজুন..."
                className="pl-10 pr-10 h-11 rounded-full bg-card border-border"
              />
              {searchQuery && (
                <button
                  onClick={() => setSearchQuery("")}
                  className="absolute right-3 top-1/2 -translate-y-1/2 text-muted-foreground hover:text-foreground"
                >
                  <X className="h-4 w-4" />
                </button>
              )}
            </div>
          </motion.div>
        </div>
      </div>

      {/* Category Filters + Products */}
      <div className="container mx-auto px-4 py-6 md:py-8">
        {/* Category Pills */}
        <div className="flex items-center gap-2 overflow-x-auto pb-4 scrollbar-hide mb-2">
          <SlidersHorizontal className="h-4 w-4 text-muted-foreground shrink-0" />
          {["All", ...categories].map((cat) => (
            <button
              key={cat}
              onClick={() => setActiveCategory(cat)}
              className={`shrink-0 px-4 py-2 rounded-full text-sm font-medium transition-all duration-200 ${
                activeCategory === cat
                  ? "bg-primary text-primary-foreground shadow-sm"
                  : "bg-muted text-muted-foreground hover:bg-muted/80 hover:text-foreground"
              }`}
            >
              {cat}
            </button>
          ))}
        </div>

        {/* Results Count */}
        <p className="text-sm text-muted-foreground mb-4">
          {filteredProducts.length}টি প্রোডাক্ট পাওয়া গেছে
          {activeCategory !== "All" && ` — ${activeCategory}`}
          {searchQuery && ` — "${searchQuery}"`}
        </p>

        {/* Product Grid */}
        <AnimatePresence mode="wait">
          {filteredProducts.length > 0 ? (
            <motion.div
              key={activeCategory + searchQuery}
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              exit={{ opacity: 0 }}
              transition={{ duration: 0.2 }}
              className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 gap-3 md:gap-4"
            >
              {filteredProducts.map((product, idx) => (
                <ShopProductCard key={product.id} product={product} index={idx} />
              ))}
            </motion.div>
          ) : (
            <motion.div
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              className="text-center py-20"
            >
              <Search className="h-12 w-12 text-muted-foreground/40 mx-auto mb-4" />
              <h3 className="font-heading font-semibold text-lg mb-1">কোনো প্রোডাক্ট পাওয়া যায়নি</h3>
              <p className="text-sm text-muted-foreground">অন্য কিওয়ার্ড বা ক্যাটাগরি দিয়ে খুঁজুন</p>
            </motion.div>
          )}
        </AnimatePresence>
      </div>

      <ShopFooter />
      <BottomBar />
    </div>
  );
};

export default Shop;
