import { createClient } from "https://esm.sh/@supabase/supabase-js@2";

const corsHeaders = {
  'Access-Control-Allow-Origin': '*',
  'Access-Control-Allow-Headers': 'authorization, x-client-info, apikey, content-type, x-supabase-client-platform, x-supabase-client-platform-version, x-supabase-client-runtime, x-supabase-client-runtime-version',
};

Deno.serve(async (req) => {
  if (req.method === 'OPTIONS') {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const { phone } = await req.json();

    if (!phone) {
      return new Response(
        JSON.stringify({ success: false, error: 'ফোন নম্বর দিন' }),
        { status: 400, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    const supabaseUrl = Deno.env.get('SUPABASE_URL')!;
    const supabaseKey = Deno.env.get('SUPABASE_SERVICE_ROLE_KEY')!;
    const supabase = createClient(supabaseUrl, supabaseKey);

    // Anti-spam: 60 sec cooldown between OTPs + max 5 per 2 hours
    const { data: lastOtp } = await supabase
      .from('password_reset_otps')
      .select('created_at')
      .eq('phone', phone)
      .order('created_at', { ascending: false })
      .limit(1)
      .maybeSingle();

    if (lastOtp) {
      const secondsSinceLast = (Date.now() - new Date(lastOtp.created_at).getTime()) / 1000;
      if (secondsSinceLast < 60) {
        const wait = Math.ceil(60 - secondsSinceLast);
        return new Response(
          JSON.stringify({ success: false, error: `${wait} সেকেন্ড পর আবার চেষ্টা করুন।` }),
          { status: 429, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
        );
      }
    }

    const twoHoursAgo = new Date(Date.now() - 2 * 60 * 60 * 1000).toISOString();
    const { data: recentOtps } = await supabase
      .from('password_reset_otps')
      .select('id')
      .eq('phone', phone)
      .gte('created_at', twoHoursAgo);

    if (recentOtps && recentOtps.length >= 5) {
      return new Response(
        JSON.stringify({ success: false, error: 'অনেকবার OTP পাঠানো হয়েছে। ২ ঘণ্টা পর আবার চেষ্টা করুন।' }),
        { status: 429, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    // Check if user exists with this phone
    const { data: profile } = await supabase
      .from('profiles')
      .select('user_id, name')
      .eq('phone', phone)
      .maybeSingle();

    if (!profile) {
      return new Response(
        JSON.stringify({ success: false, error: 'এই ফোন নম্বরে কোনো অ্যাকাউন্ট পাওয়া যায়নি' }),
        { status: 200, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    // Generate 6-digit OTP
    const otp = Math.floor(100000 + Math.random() * 900000).toString();
    const expiresAt = new Date(Date.now() + 5 * 60 * 1000); // 5 minutes

    // Invalidate old OTPs for this phone
    await supabase
      .from('password_reset_otps')
      .update({ used: true })
      .eq('phone', phone)
      .eq('used', false);

    // Store OTP
    const { error: insertError } = await supabase
      .from('password_reset_otps')
      .insert({ phone, otp, expires_at: expiresAt.toISOString() });

    if (insertError) {
      throw new Error('OTP সেভ করতে সমস্যা হয়েছে');
    }

    // Send SMS
    const smsApiKey = Deno.env.get('SMS_API_KEY');
    if (!smsApiKey) {
      throw new Error('SMS_API_KEY is not configured');
    }

    let cleanNumber = phone.replace(/[^0-9]/g, '');

    const smsMsg = `Your OTP: ${otp} (expires in 5 min)`;
    const smsUrl = `https://hakvolution.com/CAVATER/api.php?key=${smsApiKey}&number=${cleanNumber}&msg=${encodeURIComponent(smsMsg)}`;

    const smsResponse = await fetch(smsUrl);
    const smsText = await smsResponse.text();
    console.log('SMS API response:', smsResponse.status, smsText);

    return new Response(
      JSON.stringify({ success: true, message: 'OTP পাঠানো হয়েছে' }),
      { status: 200, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
    );
  } catch (error: unknown) {
    console.error('OTP send error:', error);
    const errorMessage = error instanceof Error ? error.message : 'Unknown error';
    return new Response(
      JSON.stringify({ success: false, error: errorMessage }),
      { status: 500, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
    );
  }
});
