import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from "https://esm.sh/@supabase/supabase-js@2";

const corsHeaders = {
  'Access-Control-Allow-Origin': '*',
  'Access-Control-Allow-Headers': 'authorization, x-client-info, apikey, content-type, x-supabase-client-platform, x-supabase-client-platform-version, x-supabase-client-runtime, x-supabase-client-runtime-version',
};

serve(async (req) => {
  if (req.method === 'OPTIONS') {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    // Try API key from site_settings first, fallback to env
    const supabaseUrl = Deno.env.get('SUPABASE_URL')!;
    const supabaseKey = Deno.env.get('SUPABASE_SERVICE_ROLE_KEY')!;
    const supabase = createClient(supabaseUrl, supabaseKey);

    let SMS_API_KEY = Deno.env.get('SMS_API_KEY');
    const { data: apiKeySetting } = await supabase
      .from('site_settings')
      .select('value')
      .eq('key', 'sms_api_key')
      .maybeSingle();
    if (apiKeySetting?.value) {
      SMS_API_KEY = apiKeySetting.value;
    }

    if (!SMS_API_KEY) {
      throw new Error('SMS_API_KEY is not configured');
    }

    const { number, message } = await req.json();

    if (!number || !message) {
      return new Response(
        JSON.stringify({ success: false, error: 'number and message are required' }),
        { status: 400, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    let cleanNumber = number.replace(/[^0-9]/g, '');
    console.log('Sending to number:', cleanNumber);

    const encodedMsg = encodeURIComponent(message);
    const url = `https://hakvolution.com/CAVATER/api.php?key=${SMS_API_KEY}&number=${cleanNumber}&msg=${encodedMsg}`;

    const response = await fetch(url, { method: 'GET' });
    const responseText = await response.text();
    console.log('SMS API Response:', responseText);

    return new Response(
      JSON.stringify({ success: true, response: responseText }),
      { status: 200, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
    );
  } catch (error: unknown) {
    console.error('SMS send error:', error);
    const errorMessage = error instanceof Error ? error.message : 'Unknown error';
    return new Response(
      JSON.stringify({ success: false, error: errorMessage }),
      { status: 500, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
    );
  }
});
