import { createClient } from "https://esm.sh/@supabase/supabase-js@2";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers":
    "authorization, x-client-info, apikey, content-type, x-supabase-client-platform, x-supabase-client-platform-version, x-supabase-client-runtime, x-supabase-client-runtime-version",
};

Deno.serve(async (req) => {
  if (req.method === "OPTIONS") {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const { phone_number, customer_name } = await req.json();

    if (!phone_number) {
      return new Response(JSON.stringify({ error: "Missing phone_number" }), {
        status: 400,
        headers: { ...corsHeaders, "Content-Type": "application/json" },
      });
    }

    const supabaseUrl = Deno.env.get("SUPABASE_URL")!;
    const supabaseKey = Deno.env.get("SUPABASE_SERVICE_ROLE_KEY")!;
    const supabase = createClient(supabaseUrl, supabaseKey);

    // Load AwajDigital settings from site_settings
    const { data: settings } = await supabase
      .from("site_settings")
      .select("key, value")
      .in("key", [
        "awaj_enabled",
        "awaj_api_token",
        "awaj_voice_name",
        "awaj_sender_number",
      ]);

    const settingsMap: Record<string, string> = {};
    settings?.forEach((s: any) => {
      if (s.value) settingsMap[s.key] = s.value;
    });

    // Check if voice call is enabled
    if (settingsMap.awaj_enabled !== "true") {
      return new Response(
        JSON.stringify({ success: false, message: "Voice call is disabled" }),
        { headers: { ...corsHeaders, "Content-Type": "application/json" } }
      );
    }

    // Get API token - from site_settings first, fallback to env
    const apiToken = settingsMap.awaj_api_token || Deno.env.get("AWAJ_API_KEY");
    if (!apiToken) {
      return new Response(
        JSON.stringify({ error: "AwajDigital API token not configured" }),
        { status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" } }
      );
    }

    const voiceName = settingsMap.awaj_voice_name;
    const senderNumber = settingsMap.awaj_sender_number;

    if (!voiceName || !senderNumber) {
      return new Response(
        JSON.stringify({ error: "Voice name or sender number not configured" }),
        { status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" } }
      );
    }

    // Clean phone number to 01XXXXXXXXX format
    let cleanPhone = phone_number.replace(/[^0-9]/g, "");
    if (cleanPhone.startsWith("88")) cleanPhone = cleanPhone.substring(2);
    if (!cleanPhone.startsWith("0")) cleanPhone = "0" + cleanPhone;

    // Generate a unique request_id
    const requestId = crypto.randomUUID();

    // Call AwajDigital Broadcast API (single number)
    const response = await fetch("https://api.awajdigital.com/api/broadcasts", {
      method: "POST",
      headers: {
        Authorization: `Bearer ${apiToken}`,
        Accept: "application/json",
        "Content-Type": "application/json",
      },
      body: JSON.stringify({
        request_id: requestId,
        voice: voiceName,
        sender: senderNumber,
        phone_numbers: [cleanPhone],
      }),
    });

    const data = await response.json();

    if (!response.ok) {
      console.error("AwajDigital API error:", JSON.stringify(data));
      return new Response(
        JSON.stringify({ success: false, error: data }),
        { status: response.status, headers: { ...corsHeaders, "Content-Type": "application/json" } }
      );
    }

    console.log("Voice call sent to", cleanPhone, "Response:", JSON.stringify(data));

    return new Response(
      JSON.stringify({ success: true, data }),
      { headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  } catch (err) {
    console.error("Voice call error:", err);
    return new Response(
      JSON.stringify({ error: "Internal error", details: String(err) }),
      { status: 500, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  }
});
