import { createClient } from "https://esm.sh/@supabase/supabase-js@2";

const corsHeaders = {
  'Access-Control-Allow-Origin': '*',
  'Access-Control-Allow-Headers': 'authorization, x-client-info, apikey, content-type, x-supabase-client-platform, x-supabase-client-platform-version, x-supabase-client-runtime, x-supabase-client-runtime-version',
};

Deno.serve(async (req) => {
  if (req.method === 'OPTIONS') {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const supabaseUrl = Deno.env.get('SUPABASE_URL')!;
    const supabaseKey = Deno.env.get('SUPABASE_SERVICE_ROLE_KEY')!;
    const supabase = createClient(supabaseUrl, supabaseKey);

    // Load WhatsApp settings from site_settings
    const { data: settings } = await supabase
      .from('site_settings')
      .select('key, value')
      .in('key', ['whatsapp_api_secret', 'whatsapp_account_id']);

    const settingsMap: Record<string, string> = {};
    settings?.forEach(r => { if (r.value) settingsMap[r.key] = r.value; });

    const apiSecret = settingsMap.whatsapp_api_secret;
    const accountId = settingsMap.whatsapp_account_id;

    if (!apiSecret || !accountId) {
      throw new Error('WhatsApp API secret or account ID is not configured');
    }

    const { number, message } = await req.json();

    if (!number || !message) {
      return new Response(
        JSON.stringify({ success: false, error: 'number and message are required' }),
        { status: 400, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
      );
    }

    // Clean phone number
    let cleanNumber = number.replace(/[^0-9]/g, '');
    // Add country code if not present (Bangladesh)
    if (cleanNumber.startsWith('0')) {
      cleanNumber = '88' + cleanNumber;
    } else if (!cleanNumber.startsWith('88')) {
      cleanNumber = '88' + cleanNumber;
    }
    console.log('Sending WhatsApp to:', cleanNumber);

    // SMSQuicker WhatsApp API - multipart/form-data POST
    const formData = new FormData();
    formData.append('secret', apiSecret);
    formData.append('account', accountId);
    formData.append('recipient', cleanNumber);
    formData.append('type', 'text');
    formData.append('message', message);

    const response = await fetch('https://smsquicker.com/api/send/whatsapp', {
      method: 'POST',
      body: formData,
    });

    const responseText = await response.text();
    console.log('WhatsApp API Response:', responseText);

    let responseData;
    try {
      responseData = JSON.parse(responseText);
    } catch {
      responseData = { raw: responseText };
    }

    return new Response(
      JSON.stringify({ success: true, response: responseData }),
      { status: 200, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
    );
  } catch (error: unknown) {
    console.error('WhatsApp send error:', error);
    const errorMessage = error instanceof Error ? error.message : 'Unknown error';
    return new Response(
      JSON.stringify({ success: false, error: errorMessage }),
      { status: 500, headers: { ...corsHeaders, 'Content-Type': 'application/json' } }
    );
  }
});
