import { createClient } from "https://esm.sh/@supabase/supabase-js@2";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers":
    "authorization, x-client-info, apikey, content-type, x-supabase-client-platform, x-supabase-client-platform-version, x-supabase-client-runtime, x-supabase-client-runtime-version",
};

Deno.serve(async (req) => {
  if (req.method === "OPTIONS") {
    return new Response(null, { headers: corsHeaders });
  }

  try {
    const { customer_name, customer_phone, customer_email, items, callback_url, user_id } = await req.json();

    if (!customer_name || !items || items.length === 0) {
      return new Response(JSON.stringify({ error: "Missing required fields" }), {
        status: 400,
        headers: { ...corsHeaders, "Content-Type": "application/json" },
      });
    }

    const supabaseUrl = Deno.env.get("SUPABASE_URL")!;
    const supabaseKey = Deno.env.get("SUPABASE_SERVICE_ROLE_KEY")!;
    const supabase = createClient(supabaseUrl, supabaseKey);

    // Try to get API key from site_settings first, fallback to env
    let apiKey = Deno.env.get("TZSMMPAY_API_KEY");
    const { data: apiKeySetting } = await supabase
      .from("site_settings")
      .select("value")
      .eq("key", "tzsmmpay_api_key")
      .maybeSingle();
    if (apiKeySetting?.value) {
      apiKey = apiKeySetting.value;
    }

    if (!apiKey) {
      return new Response(JSON.stringify({ error: "Payment API key not configured" }), {
        status: 500,
        headers: { ...corsHeaders, "Content-Type": "application/json" },
      });
    }

    // Calculate total amount - handle range prices like "৳450 – ৳3,800"
    let totalAmount = 0;
    for (const item of items) {
      // Extract just the first number from the price string
      const match = item.amount.match(/[\d,]+\.?\d*/);
      if (match) {
        const priceStr = match[0].replace(/,/g, "");
        totalAmount += parseFloat(priceStr) * (item.quantity || 1);
      }
    }

    // Create orders in database

    const orderRows = items.map((item: any) => ({
      customer_name,
      customer_phone: customer_phone || null,
      product_name: item.product_name,
      amount: item.amount,
      status: "pending",
      payment_method: "tzsmmpay",
      payment_status: "unpaid",
      user_id: user_id || null,
    }));

    const { data: orderData, error: orderError } = await supabase
      .from("orders")
      .insert(orderRows)
      .select("id");

    if (orderError) {
      return new Response(JSON.stringify({ error: "Failed to create order", details: orderError.message }), {
        status: 500,
        headers: { ...corsHeaders, "Content-Type": "application/json" },
      });
    }

    const orderIds = orderData.map((o: any) => o.id);
    const allProducts = items.map((i: any) => i.product_name).join(", ");

    // Email and SMS notifications are sent ONLY after payment confirmation (in tzsmmpay-verify)

    // Call TZSMMPay API with all required fields
    // Get payment gateway URL from site_settings, fallback to default
    let gatewayHost = "pay.tzssbd.com";
    const { data: gatewayUrlSetting } = await supabase
      .from("site_settings")
      .select("value")
      .eq("key", "payment_gateway_url")
      .maybeSingle();
    if (gatewayUrlSetting?.value) {
      gatewayHost = gatewayUrlSetting.value;
    }

    const paymentUrl = new URL(`https://${gatewayHost}/api/payment/create`);
    paymentUrl.searchParams.set("api_key", apiKey);
    paymentUrl.searchParams.set("amount", totalAmount.toString());
    paymentUrl.searchParams.set("currency", "BDT");
    paymentUrl.searchParams.set("cus_name", customer_name);
    paymentUrl.searchParams.set("cus_email", customer_email || "customer@example.com");
    paymentUrl.searchParams.set("cus_number", customer_phone || "01700000000");
    // Use separate paths with order IDs so callback can identify the orders
    const baseOrigin = callback_url.replace(/\/payment-callback$/, "");
    const orderIdParam = orderIds.join(",");
    paymentUrl.searchParams.set("success_url", baseOrigin + "/payment-success/" + orderIdParam);
    paymentUrl.searchParams.set("cancel_url", baseOrigin + "/payment-cancel/" + orderIdParam);
    paymentUrl.searchParams.set("callback_url", baseOrigin + "/payment-callback");

    console.log("Payment request URL:", paymentUrl.toString().replace(apiKey, "***"));
    const paymentRes = await fetch(paymentUrl.toString());
    const paymentData = await paymentRes.json();
    console.log("Payment response:", JSON.stringify(paymentData));

    if (!paymentData.payment_url) {
      // Rollback orders
      for (const id of orderIds) {
        await supabase.from("orders").delete().eq("id", id);
      }
      return new Response(JSON.stringify({ error: "Payment creation failed", details: paymentData }), {
        status: 500,
        headers: { ...corsHeaders, "Content-Type": "application/json" },
      });
    }

    // Store transaction_id in orders
    const txnId = paymentData.transaction_id || "";
    if (txnId) {
      for (const id of orderIds) {
        await supabase
          .from("orders")
          .update({ transaction_id: txnId })
          .eq("id", id);
      }
    }

    // Since gateway may strip query params on redirect, 
    // update payment_status to paid now since we'll redirect to success page
    // The actual payment verification happens asynchronously

    return new Response(
      JSON.stringify({
        payment_url: paymentData.payment_url,
        transaction_id: txnId,
        order_ids: orderIds,
      }),
      { headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  } catch (err) {
    return new Response(JSON.stringify({ error: "Internal error", details: String(err) }), {
      status: 500,
      headers: { ...corsHeaders, "Content-Type": "application/json" },
    });
  }
});
